<?php

namespace backend\models\db;
use backend\components\helpers\Formatter;
use Yii;
use yii\caching\TagDependency;


/**
 * This is the model class for table "jar_amounts".
 *
 * @property integer $id
 * @property integer $jar_id
 * @property integer $eoms_id
 * @property integer $month
 * @property integer $year
 * @property string $funds_brought_forward
 * @property string $monthly_budget_adjustment
 */
class JarAmount extends \yii\db\ActiveRecord
{
    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'jar_amounts';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['jar_id', 'month', 'year'], 'required'],
            [['jar_id', 'eoms_id', 'month', 'year'], 'integer'],
            [['funds_brought_forward', 'monthly_budget_adjustment'], 'number'],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'jar_id' => 'Jar ID',
            'eoms_id' => 'Eoms ID',
            'month' => 'Month',
            'year' => 'Year',
            'funds_brought_forward' => 'Accumulated Funds',
            'monthly_budget_adjustment' => 'Monthly Budget Adjustment'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getEoms()
    {
        return $this->hasOne(Eoms::className(), ['id' => 'eoms_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getJar()
    {
        return $this->hasOne(Jar::className(), ['id' => 'jar_id']);
    }

    /**
     * Updates funds brought forward value for the given jar and month,
     * should be used after any change to transactions/totals in the past
     * @param Jar $jar
     * @param $month
     * @param $year
     */
    public static function updateFundsBroughtForward(Jar $jar, $month, $year)
    {
        $jarAmount = JarAmount::find()
            ->where([
                'jar_id' => $jar->id,
                'month'  => $month,
                'year'   => $year
            ])
            ->one();
        if (!$jarAmount) {
            $jarAmount = new JarAmount();
            $jarAmount->setAttributes([
                'jar_id' => $jar->id,
                'month'  => $month,
                'year'   => $year
            ]);
        }
        $timeNow = strtotime($year.'-'.$month.'-20');
        $prevMonth = date('n', strtotime('-1 month', $timeNow));
        $prevYear  = date('Y', strtotime('-1 month', $timeNow));
        $jarAmount->funds_brought_forward = $jar->getMonthsFundsRemainingAmount($prevMonth, $prevYear);
        $jarAmount->save();
        TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);
    }

    /**
     * @param Eoms $eoms
     * @param float $allowance
     * @return bool
     */
    public static function adjustNextMonthsEverydayAllowance(Eoms $eoms, $allowance)
    {
        $jar = Jar::findOne([
            'type' => Jar::EVERYDAY_JAR_TYPE,
            'budget_id' => $eoms->budget_id
        ]);
        if (!$jar) {
            return false;
        }
        $nextMonthTime = strtotime('+1 month', strtotime("{$eoms->year}-{$eoms->month}-20"));
        $everydayJarAmount = JarAmount::findOne([
            'jar_id' => $jar->id,
            'month' => date('n', $nextMonthTime),
            'year' => date('Y', $nextMonthTime)
        ]);
        // clear any previous adjustment
        if ($everydayJarAmount && $everydayJarAmount->monthly_budget_adjustment) {
            $everydayJarAmount->monthly_budget_adjustment = null;
            $everydayJarAmount->save();
        }

        // don't save anything if the value isn't changed
        $nextMonthsMonthlyBudget = $jar->getMonthlyBudget(date('m', $nextMonthTime), date('Y', $nextMonthTime));
        if ($nextMonthsMonthlyBudget && Formatter::float($allowance) == Formatter::float($nextMonthsMonthlyBudget)) {
            return false;
        }

        // need to set next month everyday allowance adjustment
        if (!$everydayJarAmount) {
            $everydayJarAmount = new JarAmount();
            $everydayJarAmount->setAttributes([
                'jar_id' => $jar->id,
                'month' => date('n', $nextMonthTime),
                'year' => date('Y', $nextMonthTime)
            ]);
        }
        $everydayJarAmount->monthly_budget_adjustment = $allowance;
        return $everydayJarAmount->save();
    }
}
