<?php

namespace backend\models\db;

use backend\components\helpers\Formatter;
use Yii;

/**
 * This is the model class for table "time_slots".
 *
 * @property integer $id
 * @property integer $coach_id
 * @property string $start_date
 * @property string $end_date
 * @property string $start_time
 * @property string $end_time
 * @property bool $repeat
 * @property string $name
 * @property string $frequency
 * @property integer $repeat_every
 * @property string $repeat_until_date
 * @property integer $slot_time
 *
 * @property TimeSlotItem[] $timeSlotItems
 * @property User $coach
 */
class TimeSlot extends \yii\db\ActiveRecord
{
    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'time_slots';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['coach_id', 'start_date', 'end_date', 'start_time', 'end_time'], 'required'],
            [['coach_id', 'repeat_every', 'slot_time'], 'integer'],
            [['repeat_until_date', 'repeat', 'frequency'], 'safe'],
            [['start_time', 'end_time'], 'string', 'max' => 10],
            [['name'], 'string', 'max' => 255],
            [['frequency'], 'string', 'max' => 50],
            [['coach_id'], 'exist', 'skipOnError' => true, 'targetClass' => User::className(), 'targetAttribute' => ['coach_id' => 'id']],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'coach_id' => 'Coach ID',
            'start_date' => 'Start Date',
            'end_date' => 'End Date',
            'start_time' => 'Start Time',
            'end_time' => 'End Time',
            'name' => 'Name',
            'frequency' => 'Frequency',
            'repeat_every' => 'Repeat Every',
            'slot_time' => 'Slot Time',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTimeSlotItems()
    {
        return $this->hasMany(TimeSlotItem::className(), ['time_slot_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTimeSlotBookings()
    {
        return $this->hasMany(TimeSlotBooking::className(), ['time_slot_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getCoach()
    {
        return $this->hasOne(User::className(), ['id' => 'coach_id']);
    }

    public function generateSlotItems($localStartDateTime, $localEndDateTime)
    {
        $success = true;
        $startTime = strtotime($localStartDateTime . ':00');
        $endTime = strtotime($localEndDateTime . ':00');
        if ($this->repeat) {
            $repeatUntilTime = strtotime($this->repeat_until_date . ' 23:59:59');
            $lengthTime = $endTime - $startTime;
            $startAtTime = $startTime;
            $endAtTime = $startTime + $lengthTime;

            while ($startAtTime <= $repeatUntilTime) {
                $item = new TimeSlotItem();
                $item->setAttributes([
                    'time_slot_id' => $this->id,
                    'start_at' => Formatter::localDatetimeToUtcDatetime($startAtTime, 'Y-m-d H:i:s'),
                    'end_at' => Formatter::localDatetimeToUtcDatetime($endAtTime, 'Y-m-d H:i:s')
                ]);
                $success = $item->save() && $success;

                switch ($this->frequency) {
                    case 'daily':
                        $startAtTime = strtotime("+{$this->repeat_every} days", $startAtTime);
                        break;
                    case 'weekly':
                        $startAtTime = strtotime("+{$this->repeat_every} weeks", $startAtTime);
                        break;
                    case 'monthly':
                        $eventDay = substr($localStartDateTime, 8, 2);
                        $newEventDay = $eventDay;
                        do {
                            if ($newEventDay != $eventDay) {
                                // try again because we skipped a month (because it was shorter)
                                $startAtTime = strtotime(date(
                                    "Y-m-{$eventDay} H:i:s",
                                    strtotime("-{$newEventDay} days +{$this->repeat_every} months", $startAtTime)
                                ));
                            }
                            else {
                                $startAtTime = strtotime("+{$this->repeat_every} months", $startAtTime);
                            }
                            $newEventDay = date('d', $startAtTime);
                        }
                        while ($newEventDay != $eventDay && $startAtTime <= $repeatUntilTime);
                        break;
                    case 'annually':
                        $startAtTime = strtotime("+{$this->repeat_every} years", $startAtTime);
                        break;
                }
                $endAtTime = $startAtTime + $lengthTime;
            }
        }
        else {
            $item = new TimeSlotItem();
            $item->setAttributes([
                'time_slot_id' => $this->id,
                'start_at' => Formatter::localDatetimeToUtcDatetime($startTime, 'Y-m-d H:i:s'),
                'end_at' => Formatter::localDatetimeToUtcDatetime($endTime, 'Y-m-d H:i:s')
            ]);
            return $item->save();
        }

        return $success;
    }
}
