<?php

namespace backend\models\db;

use backend\components\helpers\Formatter;
use Yii;

/**
 * This is the model class for table "time_slot_bookings".
 *
 * @property integer $id
 * @property integer $time_slot_item_id
 * @property integer $user_id
 * @property string $start_at
 * @property string $end_at
 *
 * @property TimeSlotItem $timeSlotItem
 * @property User $user
 */
class TimeSlotBooking extends \yii\db\ActiveRecord
{
    const DEFAULT_BOOKING_MINUTES = 45;
    const CAN_CANCEL_IN_ADVANCE_HOURS = 48;


    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'time_slot_bookings';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['time_slot_id', 'time_slot_item_id', 'user_id', 'start_at', 'end_at'], 'required'],
            [['time_slot_id', 'time_slot_item_id', 'user_id'], 'integer'],
            [['start_at', 'end_at'], 'safe'],
            [['time_slot_item_id'], 'exist', 'skipOnError' => true, 'targetClass' => TimeSlotItem::className(), 'targetAttribute' => ['time_slot_item_id' => 'id']],
            [['user_id'], 'exist', 'skipOnError' => true, 'targetClass' => User::className(), 'targetAttribute' => ['user_id' => 'id']],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'time_slot_item_id' => 'Time Slot Item ID',
            'user_id' => 'User ID',
            'start_at' => 'Start At',
            'end_at' => 'End At',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTimeSlotItem()
    {
        return $this->hasOne(TimeSlotItem::className(), ['id' => 'time_slot_item_id']);
    }

    public function getUser()
    {
        return $this->hasOne(User::className(), ['id' => 'user_id']);
    }

    public function canCancelBooking()
    {
        return strtotime($this->start_at) - time() > self::CAN_CANCEL_IN_ADVANCE_HOURS * 3600;
    }

    public function getBookingInfo(TimeZone $timezone)
    {
        return Formatter::datetime(
            Formatter::utcDatetimeToLocalDatetime($this->start_at, $timezone->code,'Y-m-d H:i:s')
        ) . ' - ' . $this->user->fullName;
    }

    public function sendCreatedNotification(User $coach, TimeZone $coachTimeZone)
    {
        $this->user->addNotification('booked_appointment', [
            'name' => $coach->fullName,
            'datetime' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($this->start_at, $this->user->getTimeZone()->code,'Y-m-d H:i:s'))
        ]);
        $coach->addNotification('booked_appointment', [
            'name' => $this->user->fullName,
            'datetime' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($this->start_at, $coachTimeZone->code,'Y-m-d H:i:s'))
        ]);
    }

    public function sendCancelledNotification(User $coach, TimeZone $coachTimeZone)
    {
        $this->user->addNotification('cancelled_appointment', [
            'datetime' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($this->start_at, $this->user->getTimeZone()->code,'Y-m-d H:i:s')),
            'user_name' => $coach->fullName
        ]);
        $coach->addNotification('cancelled_appointment', [
            'datetime' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($this->start_at, $coachTimeZone->code,'Y-m-d H:i:s')),
            'user_name' => $coach->fullName
        ]);
    }

    public function sendUpcomingNotification(User $coach, TimeZone $coachTimeZone, $today)
    {
        // user
        if ($today) {
            $this->user->addNotification('appointment_today', [
                'time' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($this->start_at, $this->user->getTimeZone()->code,'Y-m-d H:i:s')),
                'name' => $coach->getFullName()
            ]);
        }
        else {
            $this->user->addNotification('appointment_tomorrow', [
                'time' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($this->start_at, $this->user->getTimeZone()->code,'Y-m-d H:i:s')),
                'name' => $coach->getFullName()
            ]);
        }
        // coach
        if ($today) {
            $coach->addNotification('appointment_today', [
                'time' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($this->start_at, $coachTimeZone->code,'Y-m-d H:i:s')),
                'name' => $this->user->getFullName()
            ]);
        }
        else {
            $coach->addNotification('appointment_tomorrow', [
                'time' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($this->start_at, $coachTimeZone->code,'Y-m-d H:i:s')),
                'name' => $this->user->getFullName()
            ]);
        }
    }


    public static function hasBookingAt($dateTime)
    {
        $startTime = strtotime($dateTime);
        $start = date('Y-m-d H:i:00', $startTime);
        $end   = date('Y-m-d H:i:00', strtotime('+' . self::DEFAULT_BOOKING_MINUTES . ' minutes', $startTime));

        return TimeSlotBooking::find()
            ->where(['or',
                ['and',
                    ['>', 'start_at', $start],
                    ['<', 'start_at', $end]
                ],
                ['and',
                    ['>', 'end_at', $start],
                    ['<', 'end_at', $end]
                ],
                ['and',
                    ['start_at' => $start],
                    ['end_at' => $end]
                ]
            ])
            ->exists();
    }

    public static function getUserBookings($userId, $startTime, $endTime, TimeZone $timezone,
                                           $constraint = false, $coachId = null)
    {
        $userEvents = TimeSlotBooking::find()
            ->where(['and',
                ['between', 'start_at', $startTime, $endTime],
                ['between', 'end_at', $startTime, $endTime]
            ]);
        if (!$coachId) {
            $userEvents = $userEvents->where([
                'user_id' => $userId
            ])->all();
        }
        else {
            if ($userId) {
                $userEvents->where(['and',
                    ['!=', 'user_id', $userId]
                ]);
            }
            $userEvents = $userEvents
                ->joinWith('timeSlotItem.timeSlot')
                ->where([
                    'coach_id' => $coachId
                ])->all();
        }
        $result = [];
        /** @var TimeSlotBooking $event */
        foreach ($userEvents as $event) {
            $e = [
                'id' => $event->id,
                'type' => TimeSlotBooking::className(),
                'allDay' => false,
                'title' => $userId ? $event->timeSlotItem->timeSlot->coach->fullName : $event->user->fullName,
                'start' => Formatter::utcDatetimeToLocalDatetime($event->start_at, $timezone->code, 'c'),
                'end' => Formatter::utcDatetimeToLocalDatetime($event->end_at, $timezone->code, 'c'),
                'editable' => $coachId ? false : true,
                'durationEditable' => false,
                'overlap' => false
            ];
            if ($constraint) {
                $e['constraint'] = 'c' . $event->timeSlotItem->id;
            }
            if ($coachId && $userId) {
                $e['rendering'] = 'background';
            }
            $result[] = $e;
        }
        return $result;
    }

    public static function getNextAppointment($userId, $dateTime = null)
    {
        if (!$dateTime) {
            $dateTime = time();
        }
        return TimeSlotBooking::find()
            ->where([
                'and',
                ['user_id' => $userId],
                ['>=', 'start_at', Formatter::localDatetimeToUtcDatetime($dateTime)]
            ])
            ->orderBy('start_at asc')
            ->one();
    }
}
