<?php

namespace backend\models\db;

use backend\components\helpers\Formatter;
use Yii;

/**
 * This is the model class for table "time_slot_items".
 *
 * @property integer $id
 * @property integer $time_slot_id
 * @property string $start_at
 * @property string $end_at
 *
 * @property TimeSlot $timeSlot
 */
class TimeSlotItem extends \yii\db\ActiveRecord
{
    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'time_slot_items';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['time_slot_id', 'start_at', 'end_at'], 'required'],
            [['time_slot_id'], 'integer'],
            [['start_at', 'end_at'], 'safe'],
            [['time_slot_id'], 'exist', 'skipOnError' => true, 'targetClass' => TimeSlot::className(), 'targetAttribute' => ['time_slot_id' => 'id']],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'time_slot_id' => 'Time Slot ID',
            'start_at' => 'Start At',
            'end_at' => 'End At',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTimeSlot()
    {
        return $this->hasOne(TimeSlot::className(), ['id' => 'time_slot_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTimeSlotBookings()
    {
        return $this->hasMany(TimeSlotBooking::className(), ['time_slot_item_id' => 'id']);
    }

    public function getPossibleHoursList(TimeZone $timezone)
    {
        $bookingLength = TimeSlotBooking::DEFAULT_BOOKING_MINUTES * 60;
        $start = strtotime($this->start_at);
        $end = strtotime($this->end_at);
        if ($end - $start < $bookingLength) {
            return [];
        }
        $hours = [];
        do {
            $utcTime = date('Y-m-d H:i:00', $start);
            if (!TimeSlotBooking::hasBookingAt($utcTime)) {
                $time = Formatter::utcDatetimeToLocalDatetime($utcTime, $timezone->code, 'Y-m-d H:i');
                $hours[$time] = substr($time, -5);
            }
            $start += 15 * 60; // allow to book every 15 minutes
        }
        while ($start + $bookingLength <= $end);
        return $hours;
    }


    public static function getTimeSlotItems($coachId, TimeZone $timezone, $fromDate = null, $asMap = true)
    {
        $fromDate = Formatter::localDatetimeToUtcDatetime(
            $fromDate ? $fromDate : date('Y-m-d')
        );

        $items = TimeSlotItem::find()
            ->joinWith('timeSlot')
            ->where(['and',
                ['coach_id' => $coachId],
                ['>=', 'DATE(start_at)', $fromDate]
            ])
            ->orderBy('start_at ASC')
            ->all();
        if ($asMap) {
            $result = [];
            $format = Yii::$app->session->get('dateFormat');
            foreach ($items as $item) {
                $result[$item->id] =
                    Formatter::utcDatetimeToLocalDatetime($item->start_at, $timezone->code, $format . ' - H:i') . ' to ' .
                    Formatter::utcDatetimeToLocalDatetime($item->end_at, $timezone->code, 'H:i') .
                    ($item->timeSlot->name ? ' (' . $item->timeSlot->name . ')' : '');
            }
            return $result;
        }
        return $items;
    }
}
