<?php

namespace backend\models\db;

use Yii;
use yii\caching\TagDependency;
use yii\helpers\Url;

/**
 * This is the model class for table "transactions".
 *
 * @property integer $id
 * @property integer $account_id
 * @property integer $expense_id
 * @property double $amount
 * @property string $date
 * @property string $description
 * @property int $checked
 * @property int $tax_deductible
 *
 * @property Account $account
 * @property Expense $expense
 * @property Jar $jar
 */
class Transaction extends \yii\db\ActiveRecord
{
    // for additional query fields
    public $name;
    public $jar_id;
    public $jar_name;
    public $expense_name;
    public $parent_name;
    public $account_name;
    public $is_adjustment;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'transactions';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['account_id', 'expense_id', 'checked', 'tax_deductible'], 'integer'],
            ['amount', 'number'],
            ['date', 'safe'],
            ['description', 'string', 'max' => 255],

            [['account_id', 'amount', 'date'], 'required', 'on' => 'api'],
            [['description', 'jar_id', 'expense_id', 'checked'], 'safe', 'on' => 'api']
        ];
    }

    public function fields()
    {
        $fields = parent::fields();

        $fields['account_name'] = function($model) { return $model->account_name; };
        $fields['expense_name'] = function($model) { return $model->name; };
        $fields['jar_id'] = function($model) {
            if ($model->jar_id) {
                return $model->jar_id;
            }
            $jar = $model->expense->jar;
            if ($jar) {
                return $jar->id;
            }
            return null;
        };
        $fields['jar_name'] = function($model) {
            if ($model->jar_name) {
                return $model->jar_name;
            }
            $jar = $model->expense->jar;
            if ($jar) {
                return $jar->name;
            }
            return null;
        };
        $fields['type'] = function() { return 'expense'; };

        return $fields;
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'account_id' => 'Account ID',
            'expense_id' => 'Expense ID',
            'amount' => 'Amount',
            'date' => 'Date',
            'checked' => 'Checked'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccount()
    {
        return $this->hasOne(Account::className(), ['id' => 'account_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getExpense()
    {
        return $this->hasOne(Expense::className(), ['id' => 'expense_id']);
    }

    /**
     * @inheritdoc
     */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);

        // need to have a jar_id defined
        if (empty($this->expense->jar_id)) {
            return;
        }

        TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $this->expense->jar_id);

        // check for jar overspending
        $jar = $this->expense->jar;
        $jarBalance = $jar->getThisMonthsFundsRemainingAmount();

        if ($jarBalance < 0) {
            // check if already notified
            $notification = Notification::find()
                ->leftJoin(NotificationData::tableName(), "`key` = 'jar_id' AND `value` = :jar_id", ['jar_id' => $jar->id])
                ->where([
                    'AND',
                    ['MONTH(created_at)' => date('m')],
                    ['YEAR(created_at)' => date('Y')],
                    ['notification_template_name' => 'jar_overspent'],
                    ['IS NOT', 'value', null]
                ])
                ->one();

            if (!$notification && !$jar->budget->is_mock) {
                $userId = UserMeta::find()
                   ->select('user_id')
                   ->where([
                       'AND',
                       ['key' => 'active_budget_id'],
                       ['value' => $jar->budget->id]
                   ])
                   ->scalar();

                $user = User::findOne($userId);
                if ($user) {
                    $newNotification = $user->addNotification('jar_overspent', [
                        'url' => Url::toRoute(['/jars/expense-jar-detail', 'id' => $jar->id], true),
                        'name' => $jar->name
                    ]);

                    $data = new NotificationData();
                    $data->setAttributes([
                        'key' => 'jar_id',
                        'value' => (string)$jar->id,
                        'notification_id' => $newNotification->id
                    ]);
                    $data->save();
                }
            }
        }
    }

    /**
     * @inheritdoc
     */
    public function afterDelete()
    {
        if ($this->expense->jar_id) {
            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $this->expense->jar_id);
        }
        parent::afterDelete();
    }


    /**
     * Get the jar this transaction belongs to.
     * @return ActiveQuery object.
     */
    public function getJar() {

        $expense = $this->getExpense();
        return $expense->getJar();

    }

}
