<?php

namespace backend\models\db;

use Yii;
use backend\models\ChangeMetaModel;

use backend\components\helpers\Calculator;

/**
 * This is the model class for table "transfers".
 *
 * @property integer $id
 * @property integer $account_from_id
 * @property integer $account_to_id
 * @property string $name
 * @property string $frequency
 * @property double $amount
 * @property string $date
 * @property bool $is_adjustment
 * @property integer $archived
 * @property integer $budget_id
 *
 * @property TransferChange[] $transferChanges
 * @property TransferEvent[] $transferEvents
 * @property Account $accountFrom
 * @property Account $accountTo
 */
class Transfer extends ChangeMetaModel
{

    public $account_from_name;
    public $account_to_name;

    private $_nextDue;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'transfers';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['account_from_id', 'account_to_id', 'archived', 'budget_id'], 'integer'],
            [['amount'], 'number'],
            [['date'], 'safe'],
            [['name'], 'string', 'max' => 200],
            [['frequency'], 'string', 'max' => 20],

            [['account_from_id', 'account_to_id', 'name', 'amount', 'date'], 'required', 'on' => 'api']
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'budget_id' => 'Budget ID',
            'account_from_id' => 'Account From ID',
            'account_to_id' => 'Account To ID',
            'name' => 'Name',
            'frequency' => 'Frequency',
            'amount' => 'Amount',
            'date' => 'Date',
            'archived' => 'Archived',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTransferChanges()
    {
        return $this->hasMany(TransferChange::className(), ['transfer_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTransferEvents()
    {
        return $this->hasMany(TransferEvent::className(), ['transfer_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccountFrom()
    {
        return $this->hasOne(Account::className(), ['id' => 'account_from_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccountTo()
    {
        return $this->hasOne(Account::className(), ['id' => 'account_to_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTransferEnd()
    {
        return $this->hasOne(TransferEnd::className(), ['transfer_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBudget()
    {
        return $this->hasOne(Budget::className(), ['id' => 'budget_id']);
    }

    /**
     * @inheritdoc
     */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        if ($this->scenario == self::SCENARIO_CLONE) {
            return;
        }

        // transfer created
        if ($insert) {
            // generate transaction if one-time OR today or in past (not for adjustments)
            if (!$this->is_adjustment &&
                ($this->frequency == 'one-time' || $this->date <= date('Y-m-d'))) {
                $transferEvent = new TransferEvent();
                $transferEvent->setAttributes([
                    'transfer_id' => $this->id,
                    'amount'      => $this->amount,
                    'date'        => $this->date
                ]);
                $transferEvent->save();
            }

            // save the change
            $transferChange = new TransferChange();
            $transferChange->setAttributes([
                'type'        => 'C',
                'transfer_id' => $this->id,
                'user_id'     => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $transferChange->save();
        }
        // transfer edited
        else {
            // check if and what changed
            $changes = array();
            $transferAttrs = $this->getAttributes();

            foreach ($changedAttributes as $key => $o) {
                if (isset($transferAttrs[$key]) && ($transferAttrs[$key] != $o)) {
                    $changes[$key] = $o;
                }
            }

            if (count($changes) > 0) {
                $transferChange = new TransferChange();
                $transferChange->setAttributes([
                    'type'        => 'E',
                    'transfer_id' => $this->id,
                    'user_id'     => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                        Yii::$app->user->identity->id : null
                ]);
                $transferChange->save();

                foreach($changes as $key => $value) {
                    $transferChangeMeta = new TransferChangeMeta();
                    $transferChangeMeta->setAttributes([
                        'transfer_change_id' => $transferChange->id,
                        'key'                => $key,
                        'value'              => (string)$value
                    ]);
                    $transferChangeMeta->save();
                }
            }
        }
        // transfer removed
        if (isset($changedAttributes['archived']) && $this->archived == 1) {
            $transferChange = new TransferChange();
            $transferChange->setAttributes([
                'type'        => 'D',
                'transfer_id' => $this->id,
                'user_id'     => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $transferChange->save();
        }
    }

    /**
    * Get the next due date of transfer event.
    * @return string Date of next due date of debt payment.
    */
    public function getNextDue($startDate = false, $reset = false, $dateLimit = false)
    {
        if (($this->_nextDue == null) || $reset) {
            if ($startDate) {
                $this->_nextDue = Calculator::nextDueDate($this->date, $this->frequency, $dateLimit, $startDate);
            }
            else {
                $this->_nextDue = Calculator::nextDueDate($this->date, $this->frequency, $dateLimit);
            }
        }
        return $this->_nextDue;
    }

    public function deleteCompletely()
    {
        TransferEnd::deleteAll(['transfer_id' => $this->id]);
        $this->delete();
    }


    public static function getTransfersList($budgetId)
    {
        return Transfer::find()
            ->where([
                'and',
                ['archived' => 0],
                ['is_adjustment' => 0],
                ['budget_id' => $budgetId],
                ['!=', 'frequency', 'one-time']
            ])
            ->orderBy('name ASC')
            ->all();
    }
}
