<?php

namespace backend\models\db;

use backend\components\helpers\DripHelper;
use backend\components\helpers\Formatter;
use backend\components\helpers\SharpSpringHelper;
use backend\models\ChangeMetaModel;
use common\models\db\UserPasswordResetRequest;
use DrewM\MailChimp\MailChimp;
use Imagine\Image\Box;
use Imagine\Image\Point;
use Yii;
use yii\base\NotSupportedException;
use yii\helpers\Url;
use yii\imagine\Image;
use yii\web\IdentityInterface;
use yii\web\UploadedFile;

/**
 * User model
 *
 * @property integer $id
 * @property integer $user_role_id
 * @property string $username
 * @property string $password_hash
 * @property string $password_salt
 * @property string $password_reset_token
 * @property string $email
 * @property string $auth_key
 * @property integer $archived
 * @property string $password write-only password
 * @property string $click_funnels_data
 */
class User extends ChangeMetaModel implements IdentityInterface {

    const STATUS_DELETED   = 1;
    const STATUS_ACTIVE    = 0;

    const RESULT_INCORRECT = 1;
    const RESULT_INACTIVE  = 2;
    const RESULT_FTS       = 3;
    const RESULT_ERROR     = 4;

    public $first_name;
    public $last_name;
    public $meta;
    public $user_parent_id;
    public $login_time;
    public $login_duration;
    public $seconds_ago;
    public $unread_messages;

    private $_clients;

    /**
     * @inheritdoc
     */
    public static function tableName() {
        return 'users';
    }

    /**
     * @inheritdoc
     */
    public function rules() {
        return [
            ['archived', 'default', 'value' => self::STATUS_ACTIVE],
            ['archived', 'in', 'range' => [self::STATUS_ACTIVE, self::STATUS_DELETED]],
            [['username', 'user_role_id', 'email'], 'required'],
            ['email', 'email'],
        ];
    }

    public function fields() {
        $fields = parent::fields();
        unset($fields['password'], $fields['password_salt'], $fields['auth_key']);

        // add user meta keys
        foreach (UserMeta::METAKEYS as $keyName => $name) {
            switch ($keyName) {
                case 'country_id':
                    $key = 'country';
                    $function = function($model) use ($keyName) {
                        $result = Country::findOne(
                            $model->getMetaValueByKey($keyName)
                        );
                        return $result ? $result->name : '';
                    };
                    break;
                case 'date_format_id':
                    $key = 'date_format';
                    $function = function($model) use ($keyName) {
                        return DateFormat::getDateFormat(
                            $model->getMetaValueByKey($keyName)
                        );
                    };
                    break;
                case 'time_format_id':
                    $key = 'time_format';
                    $function = function($model) use ($keyName) {
                        return TimeFormat::getTimeFormat(
                            $model->getMetaValueByKey($keyName)
                        );
                    };
                    break;
                case 'time_zone_id':
                    $key = 'time_zone';
                    $function = function($model) use ($keyName) {
                        $result = TimeZone::getTimeZone(
                            $model->getMetaValueByKey($keyName)
                        );
                        return $result ? $result->name : '';
                    };
                    break;
                case 'currency_id':
                    $key = 'currency';
                    $function = function($model) use ($keyName) {
                        $result = Currency::getCurrency(
                            $model->getMetaValueByKey($keyName)
                        );
                        return $result ? $result->code : '';
                    };
                    break;
                case 'currency_format':
                    $key = 'currency_format';
                    $function = function($model) use ($keyName) {
                        return Currency::getCurrencyFormatDetails(
                            $model->getMetaValueByKey($keyName)
                        );
                    };
                    break;
                default:
                    $key = $keyName;
                    $function = function($model) use ($key) {
                        return $model->getMetaValueByKey($key);
                    };
                    break;
            }
            $fields[$key] = $function;
        }

        return $fields;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAppointments() {
        return $this->hasMany(Appointment::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAppointmentUsers() {
        return $this->hasMany(AppointmentUser::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBudgetChats() {
        return $this->hasMany(BudgetChat::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBudget() {
        return $this->hasOne(Budget::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBudgets() {
        return $this->hasMany(Budget::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBugs() {
        return $this->hasMany(Bug::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getConversationMessages() {
        return $this->hasMany(ConversationMessage::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getConversationsToUsers() {
        return $this->hasMany(ConversationsToUser::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getConversations() {
        return $this->hasMany(Conversation::className(), ['id' => 'conversation_id'])->viaTable('conversations_to_users', ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getEmails() {
        return $this->hasMany(Email::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getNotes() {
        return $this->hasMany(Note::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getClientNotes() {
        return $this->hasMany(Note::className(), ['user_client_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getNotifications() {
        return $this->hasMany(Notification::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserLogins() {
        return $this->hasMany(UserLogin::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserMeta() {
        return $this->hasOne(UserMeta::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserMetas() {
        return $this->hasMany(UserMeta::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserParent() {
        return $this->hasOne(UserParent::className(), ['user_id' => 'id']);
    }

    public function getFirstNameMeta() {
        return $this->hasOne(UserMeta::className(), ['user_id' => 'id'])
            ->alias('fnm')
            ->where([
                'fnm.key' => UserMeta::FIRST_NAME
            ]);
    }

    public function getLastNameMeta() {
        return $this->hasOne(UserMeta::className(), ['user_id' => 'id'])
            ->alias('lnm')
            ->where([
                'lnm.key' => UserMeta::LAST_NAME
            ]);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getParent() {
        return $this->hasOne(User::className(), ['id' => 'user_id'])->via('userParent');
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserChildren()
    {
        return $this->hasMany(UserParent::className(), ['user_parent_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getChildren()
    {
        return $this->hasMany(User::className(), ['id' => 'user_parent_id'])->via('userChildren');
    }


    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserPasswordResetRequests() {
        return $this->hasMany(UserPasswordResetRequest::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserPlans() {
        return $this->hasMany(UserPlan::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserRegistrations() {
        return $this->hasMany(UserRegistration::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserRole() {
        return $this->hasOne(UserRole::className(), ['id' => 'user_role_id']);
    }

    /**
     * @inheritdoc
     */
    public static function findIdentity($id) {
        return static::findOne(['id' => $id, 'archived' => self::STATUS_ACTIVE]);
    }

    /**
     * @inheritdoc
     */
    public static function findIdentityByAccessToken($token, $type = null) {
        throw new NotSupportedException('"findIdentityByAccessToken" is not implemented.');
    }

    /**
     * @inheritdoc
     */
    public function getId() {
        return $this->getPrimaryKey();
    }

    /**
     * @inheritdoc
     */
    public function getAuthKey() {
        return $this->auth_key;
    }

    /**
     * @inheritdoc
     */
    public function validateAuthKey($authKey) {
        return $this->getAuthKey() === $authKey;
    }

    public function attributeLabels() {
        return [
            'user_role_id' => 'User Role',
            'passwordText' => 'Password',
        ];
    }

    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        if ($this->scenario == self::SCENARIO_CLONE) {
            return;
        }

        if ($insert) {
            // setting default meta values for the client
            if ($this->user_role_id == UserRole::CLIENT || $this->user_role_id == UserRole::COACH) {
                foreach (UserMeta::getDefaultUserMeta() as $key => $value) {
                    $meta = new UserMeta();
                    $meta->setAttributes([
                        'user_id' => $this->id,
                        'key'     => $key,
                        'value'   => (string)$value
                    ]);
                    $meta->save();
                }
            }

            // assign user rbac role
            $this->assignRole();

            // save change info
            $userChange = new UserChange();
            $userChange->setAttributes([
                'type'           => 'C',
                'user_record_id' => $this->id,
                'user_id'        => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $userChange->save();
        }
        // user record removed
        elseif (count($changedAttributes) == 1 &&
            isset($changedAttributes['archived']) &&
            $this->archived == 1) {
            $userChange = new UserChange();
            $userChange->setAttributes([
                'type'           => 'D',
                'user_record_id' => $this->id,
                'user_id'        => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $userChange->save();
        }
        // user record updated
        else {
            // check if and what changed
            $changes = array();
            $userAttrs = $this->getAttributes();

            foreach ($changedAttributes as $key => $o) {
                if (isset($userAttrs[$key]) && ($userAttrs[$key] != $o)) {
                    $changes[$key] = $o;
                }
            }

            if (count($changes) > 0) {
                $userChange = new UserChange();
                $userChange->setAttributes([
                    'type'           => 'E',
                    'user_record_id' => $this->id,
                    'user_id'        => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                        Yii::$app->user->identity->id : null
                ]);
                $userChange->save();

                foreach($changes as $key => $value) {
                    $userChangeMeta = new UserChangeMeta();
                    $userChangeMeta->setAttributes([
                        'user_change_id' => $userChange->id,
                        'key'            => $key,
                        'value'          => (string)$value
                    ]);
                    $userChangeMeta->save();
                }
            }
        }
    }


    /**
     * Get  a sorted array of user meta.
     * @return array User meta.
     */
    public function getUserMetaSorted() {

        if ($this->meta == null) {

            $meta = $this->userMetas;
            $this->meta = [];
            foreach ($meta as $m) {

                $this->meta[$m->key] = $m->value;

            }

            // let's add a full name
            $this->meta['full_name'] = '';
            if (isset($this->meta['first_name'])) {

                $this->meta['full_name'] .= $this->meta['first_name'];

            }

            if (isset($this->meta['last_name']) && (strlen($this->meta['last_name']) > 0)) {

                if (strlen($this->meta['full_name'])) {

                    $this->meta['full_name'] .= ' ';

                }

                $this->meta['full_name'] .= $this->meta['last_name'];

            }

            // let's also add a single-line address
            $addressLines = [];
            if (isset($this->meta['address_line_1']) && (strlen($this->meta['address_line_1']) > 0)) {

                $addressLines[] = $this->meta['address_line_1'];

            }
            if (isset($this->meta['address_line_2']) && (strlen($this->meta['address_line_2']) > 0)) {

                $addressLines[] = $this->meta['address_line_2'];

            }
            if (isset($this->meta['address_line_3']) && (strlen($this->meta['address_line_3']) > 0)) {

                $addressLines[] = $this->meta['address_line_3'];

            }
            $this->meta['address_one_line'] = implode(', ', $addressLines);

            // aaand a country
            $this->meta['country_name'] = '';

            if (isset($this->meta['country_id']) && (strlen($this->meta['country_id']) > 0)) {

                $this->meta['country_name'] = Country::findOne($this->meta['country_id'])->name;

            }

        }

        return $this->meta;

    }

    /**
     * Get the value of a certain key from UserMeta table. If a first and last names are not found, the function
     * fallsback to username.
     * @param string $key
     * @return bool|string
     * @throws \Exception
     */
    public function getMetaValueByKey($key)
    {
        if (!in_array($key, UserMeta::getMetaKeys())) {
            throw new \Exception('Cannot find a value for "' . $key . '"');
        }

        $meta = $this->getUserMetas()
            ->select(['value'])
            ->andWhere(['key' => $key])
            ->one();
        return $meta ? $meta->value : false;
    }

    /**
     * Assigns auth role
     */
    public function assignRole()
    {
        $auth = Yii::$app->authManager;
        $userRole = $this->getUserRole()->one()->name;
        if ($userRole == 'subClient') {
            $userRole = 'sub-client';
        }
        $role = $auth->getRole($userRole);
        $auth->assign($role, $this->id);
    }

    public function getFirstName()
    {
        if ($this->first_name == null) {
            $this->first_name = $this->getMetaValueByKey(UserMeta::FIRST_NAME);
        }
        return $this->first_name;
    }

    public function getLastName()
    {
        if ($this->last_name == null) {
            $this->last_name = $this->getMetaValueByKey(UserMeta::LAST_NAME);
        }
        return $this->last_name;
    }

    /**
     * Query the first and last name of the user an return them in a string.
     * @param bool $appendUsername
     * @return string
     */
    public function getFullName($appendUsername = false)
    {
        $fullName = trim($this->getFirstName() . ' ' . $this->getLastName());
        if (empty($fullName)) {
            return $this->username;
        }
        elseif ($appendUsername) {
            $fullName .= ' (' . $this->username . ')';
        }
        return $fullName;
    }

    /**
     * @param static|int $user
     * @param string $message
     * @return bool
     */
    public function sendMessageTo($user, $message)
    {
        return Conversation::sendMessage($this->id, $user, $message);
    }

    /**
     * @return bool|User
     * @throws \Exception
     */
    public function getCoach()
    {
        $parentUser = $this->userParent->userParent;
        if ($this->user_role_id == UserRole::CLIENT) {
            $coach = $parentUser;
        }
        elseif ($this->user_role_id == UserRole::SUBCLIENT) {
            $coach = $parentUser->userParent->userParent;
        }
        return (isset($coach) && $coach) ? $coach : false;
    }

    public function getCoachUsers()
    {
        if ($this->user_role_id != UserRole::COACH) {
            return [];
        }
        return User::find()
            ->joinWith(['userParent', 'lastNameMeta', 'firstNameMeta'])
            ->where([
                'user_parent_id' => $this->id,
                'archived' => 0,
                'user_role_id' => UserRole::CLIENT
            ])
            ->orderBy([
                'lnm.value' => 'ASC',
                'fnm.value' => 'ASC'
            ])
            ->all();
    }

    public function canBookAppointment()
    {
        return !TimeSlotBooking::find()->where(['and',
            ['user_id' => $this->id],
            ['>', 'start_at', date('Y-m-d H:i:s')]
        ])->exists();
    }

    /**
     * Generates "remember me" authentication key
     */
    public function generateAuthKey() {
        $this->auth_key = Yii::$app->security->generateRandomString();
    }

    /**
     * Determines if the user shares the email with another user (subaccount or parent account)
     * @return bool
     */
    public function hasNotUniqueEmail()
    {
        if ($this->user_role_id == UserRole::CLIENT) {
            foreach ($this->getUserChildren()->all() as $subClient) {
                if (stripos($this->email, $subClient->user->email) === 0) {
                    return true;
                }
            }
        }
        else {
            $parent = $this->userParent;
            if ($parent && $parent->userParent && stripos($parent->userParent->email, $this->email) === 0) {
                return true;
            }
        }
        return false;
    }

    /**
     * Tells if the user's email is unique in the entire database
     * @return bool
     */
    public function uniqueEmailInDatabase()
    {
        return !User::find()
            ->where(['email' => $this->email])
            ->andWhere(['!=', 'id', $this->id])
            ->exists();
    }

    /*public function sharpSpringRegistered(UserPlan $userPlan)
    {
        if (!isset(Yii::$app->params['sharpspring'])) {
            return false;
        }
        $sharpSpring = new SharpSpringHelper(
            Yii::$app->params['sharpspring']['account_id'],
            Yii::$app->params['sharpspring']['secret_key']
        );
        $attributes = [
            'firstName' => $this->getFirstName(),
            'lastName' => $this->getLastName(),
            'Plan Name' => $userPlan->userPlanType->name,
            'Verified Email' => false
        ];
        // for Bootcamp users
        if (stripos($userPlan->userPlanType->name, 'bootcamp') !== false ||
            stripos($userPlan->userPlanType->name, 'free') !== false) {
            return $sharpSpring->createOrUpdateLead($this->email, $attributes, [
                SharpSpringHelper::LIST_LEADS
            ]);
        }
        else {
            return $sharpSpring->createOrUpdateLead($this->email, $attributes, [
                SharpSpringHelper::LIST_CUSTOMERS
            ]);
        }
    }*/

    /*public function sharpSpringVerified(UserPlan $userPlan = null, $verified = true)
    {
        if (!isset(Yii::$app->params['sharpspring'])) {
            return false;
        }
        $sharpSpring = new SharpSpringHelper(
            Yii::$app->params['sharpspring']['account_id'],
            Yii::$app->params['sharpspring']['secret_key']
        );
        return $sharpSpring->createOrUpdateLead($this->email, [
            'firstName' => $this->getFirstName(),
            'lastName' => $this->getLastName(),
            'Plan Name' => $userPlan->userPlanType->name,
            'Verified Email' => $verified
        ]);
    }*/

    /*public function sharpSpringChangedPlan(UserPlan $userPlan)
    {
        if (!isset(Yii::$app->params['sharpspring'])) {
            return false;
        }
        $sharpSpring = new SharpSpringHelper(
            Yii::$app->params['sharpspring']['account_id'],
            Yii::$app->params['sharpspring']['secret_key']
        );
        return $sharpSpring->createOrUpdateLead($this->email, [
            'firstName' => $this->getFirstName(),
            'lastName' => $this->getLastName(),
            'Plan Name' => $userPlan->userPlanType->name
        ], [
            SharpSpringHelper::LIST_CUSTOMERS
        ]);
    }*/

    public function mailchimpSubscribe()
    {
        if (!isset(Yii::$app->params['mailchimp'])) {
            return false;
        }
        $mailchimp = new MailChimp(Yii::$app->params['mailchimp']['api-key']);
        return $mailchimp->post('lists/' . Yii::$app->params['mailchimp']['list-id'] . '/members', [
            'email_address' => $this->email,
            'status' => 'subscribed',
            'merge_fields' => [
                'FNAME' => $this->getFirstName()
            ]
        ]);
    }

    public function isMembershipUser(array $membersList)
    {
        foreach ($membersList as $member) {
            $formattedUsername = strtr(strtolower($this->username), [
                '.' => '-',
                '&' => '-'
            ]);
            if (isset($member['slug']) && strtolower($member['slug']) == $formattedUsername) {
                return explode(',', $member['ihc_user_levels']);
            }
        }
        return false;
    }

    public function createAndSaveMembershipUser($password, $firstName = null, $lastName = null)
    {
        if (!isset(Yii::$app->getComponents()['siteConnector'])) {
            return false;
        }
        $userPlan = UserPlan::getActivePlan($this->id);
        $attributes = [
            'username' => strtr($this->username, ['&' => '-']),
            'email' => $this->email,
            'password' => $password,
            'trial_start_date' => $userPlan ? substr($userPlan->start_time, 0, 10) : '',
            'user_plan_id' => $userPlan ? (string)$userPlan->id : 0
        ];
        if ($firstName && $lastName) {
            $attributes['first_name'] = $firstName;
            $attributes['last_name'] = $lastName;
            $attributes['name'] = $firstName . ' ' . $lastName;
        }
        $metaKey = UserMeta::findOne([
            'user_id' => $this->id,
            'key' => UserMeta::MEMBERSHIP_USER
        ]);

        $levels = $userPlan ? $userPlan->toMembershipLevel() : '1';
        $membershipUser = Yii::$app->siteConnector->createMembershipUser($attributes, $levels);
        if (!$membershipUser) {
            if (!$metaKey) {
                return false;
            }
            $membershipUser = Yii::$app->siteConnector->updateMembershipUserDetails($metaKey->value, $attributes, $levels);
        }

        if ($membershipUser) {
            if (!$metaKey) {
                $metaKey = new UserMeta();
                $metaKey->setAttributes([
                    'user_id' => $this->id,
                    'key' => UserMeta::MEMBERSHIP_USER
                ]);
            }
            $metaKey->value = (string)$membershipUser['id'];
            return $metaKey->save();
        }
        return false;
    }

    public function updateMembershipUser($password = null, $firstName = null, $lastName = null)
    {
        if (!isset(Yii::$app->getComponents()['siteConnector'])) {
            return false;
        }
        $metaKey = UserMeta::findOne([
            'user_id' => $this->id,
            'key' => UserMeta::MEMBERSHIP_USER
        ]);
        if (!$metaKey) {
            return false;
        }
        $attributes = [
            'slug' => $this->username,
            'email' => $this->email
        ];
        /*if ($password) {
            $attributes['password'] = $password;
        }*/
        if ($firstName && $lastName) {
            $attributes['first_name'] = $firstName;
            $attributes['last_name'] = $lastName;
            $attributes['name'] = $firstName . ' ' . $lastName;
        }
        return Yii::$app->siteConnector->updateMembershipUserDetails($metaKey->value, $attributes);
    }

    public function updateMembershipLevel(UserPlan $userPlan)
    {
        if (!isset(Yii::$app->getComponents()['siteConnector'])) {
            return false;
        }
        $metaKey = UserMeta::findOne([
            'user_id' => $this->id,
            'key' => UserMeta::MEMBERSHIP_USER
        ]);
        if (!$metaKey) {
            return false;
        }

        return Yii::$app->siteConnector->updateMembershipUserDetails($metaKey->value, [
            'ihc_user_levels' => $userPlan->toMembershipLevel()
        ]);
    }

    public function removeMembershipLevel(UserPlan $userPlan)
    {
        if (!isset(Yii::$app->getComponents()['siteConnector'])) {
            return false;
        }
        $metaKey = UserMeta::findOne([
            'user_id' => $this->id,
            'key' => UserMeta::MEMBERSHIP_USER
        ]);
        if (!$metaKey) {
            return false;
        }

        $member = Yii::$app->siteConnector->getUserByUsername($this->username);
        if ($member) {
            $resultLevels = [];
            $currentLevels = explode(',', $member['ihc_user_levels']);
            $removeLevels = explode(',', $userPlan->toMembershipLevel());
            foreach ($currentLevels as $level) {
                if (!in_array($level, $removeLevels)) {
                    $resultLevels[] = $level;
                }
            }
            if (count($resultLevels) != count($currentLevels)) {
                return Yii::$app->siteConnector->updateMembershipUserDetails($metaKey->value, [
                    'ihc_user_levels' => count($resultLevels) > 0 ?
                        implode(',', $resultLevels) : ''
                ]);
            }
        }
        return false;
    }

    public function deleteMembershipUser()
    {
        if (!isset(Yii::$app->getComponents()['siteConnector'])) {
            return false;
        }
        $metaKey = UserMeta::findOne([
            'user_id' => $this->id,
            'key' => UserMeta::MEMBERSHIP_USER
        ]);
        if (!$metaKey) {
            return false;
        }
        return Yii::$app->siteConnector->deleteMembershipUserDetails($metaKey->value);
    }

    /**
     * Creates and returns a new reset password request record
     * @param string|null $source
     * @return UserPasswordResetRequest
     */
    public function createNewPasswordResetRequest($source = null)
    {
        // delete all old requests for the user
        UserPasswordResetRequest::deleteAll([
            'user_id' => $this->id
        ]);

        $request = new UserPasswordResetRequest();
        $request->setAttributes([
            'user_id' => $this->id,
            'auth_token' => Yii::$app->security->generateRandomString(),
            'source' => $source
        ]);
        $request->save();
        return $request;
    }

    /**
     * Finds user by username
     *
     * @param string $username
     * @return static|null
     */
    public static function findByUsername($username)
    {
        return static::findOne(['username' => $username, 'archived' => self::STATUS_ACTIVE]);
    }

    /**
     * Finds user by password reset token
     *
     * @param string $token password reset token
     * @return static|null
     */
    public static function findByPasswordResetToken($token)
    {
        $request = UserPasswordResetRequest::find()
            ->joinWith('user')
            ->where(['and',
                ['auth_token' => $token],
                ['>=', 'time', date('Y-m-d H:i:s', time() - UserPasswordResetRequest::REQUEST_EXPIRY_TIME)],
                ['used_time' => null],
                ['archived' => self::STATUS_ACTIVE]
            ])
            ->one();
        if ($request) {
            return $request->getUser()->one();
        }
        return null;
    }

    /**
     * Return start time of User Plan
     * @param  boolean      $firstPlan determines the User Plan, true: the first plan, false (default) the last plan
     * @return date-time    start time
     */
    public function getPlanFirstDate($firstPlan = false) {
        if ($firstPlan) {
            $order = SORT_ASC;
        } else {
            $order = SORT_DESC;
        }

        $userPlan_first = UserPlan::find()
                            ->where(['user_id' => Yii::$app->user->identity->id])
                            ->orderBy(['start_time' => $order])
                            ->limit(1)
                            ->one();

        return date_create($userPlan_first->start_time);
    }

    /**
     * Return expiration time of User Plan
     * @param  boolean      $firstPlan determines the User Plan, true: the first plan, false (default) the last plan
     * @return date-time    expiration time
     */
    public function getPlanLastDate($firstPlan = false) {
        if ($firstPlan) {
            $order = SORT_ASC;
        } else {
            $order = SORT_DESC;
        }

        $userPlan_last = UserPlan::find()
                            ->where(['user_id' => Yii::$app->user->identity->id])
                            ->orderBy(['start_time' => $order])
                            ->limit(1)
                            ->one();

        return date_create(date('Y-m-d H:i:s', $userPlan_last->getExpirationTime()));
    }

    /**
     * @inheritdoc
     * Get a list of users who have the main user as a parent or are the main user.
     *
     * ->meta has additional values in addition to the default meta, like "full_name" and "address_one_line" and "country_name"
     *
     * @param int $mainUserId ID of the main user.
     * @param bool $includeArchived Whether or not to include archived users.
     * @return array List of users.
     */
    public static function getAllUsersForUserId($mainUserId, $includeArchived = false) {

        $usersQuery = self::find()
            ->select(self::tableName() . '.* , ' . UserParent::tableName() . '.user_parent_id')
            ->leftJoin(UserParent::tableName(), self::tableName() . '.id = ' . UserParent::tableName() . '.user_id')
            ->where([
                'or',
                ['user_parent_id' => $mainUserId],
                ['id' => $mainUserId]
            ]);

        if (!$includeArchived) {

            $usersQuery->andWhere(['archived' => '0']);

        }

        $users = $usersQuery->orderBy('user_parent_id ASC, username ASC')
            ->all();

        foreach ($users as $u) {

            $u->getUserMetaSorted();

        }

        return $users;
    }


    /**
     * Add a new budget for the current user.
     * @param string $name Name of the new budget.
     * @return bool|Budget Success.
     */
    public function addNewBudget($name)
    {
        // check if is main user
        if ($this->user_role_id != UserRole::CLIENT) {
            return false;
        }
        else {
            $budget = new Budget();
            $budget->setAttributes([
                'user_id' => $this->id,
                'name' => $name
            ]);
            $budget->save();
            return $budget->setUpInitialEntities();
        }
    }

    /**
     * This function just resets all the data on the user's account -
     * deleting any transactions, expenses, accounts, etc and
     * starting totally fresh, just as the user has just registered
     *
     * @param bool $reinit
     */
    public function resetAccount($reinit = true)
    {
        foreach ($this->budgets as $budget) {
            foreach ($budget->jars as $jar) {
                foreach ($jar->expenses as $expense) {
                    $expense->delete();
                }
                foreach ($jar->incomes as $income) {
                    $income->delete();
                }
                $jar->delete();
            }
            foreach ($budget->incomes as $income) {
                $income->delete();
            }
            foreach ($budget->debts as $debt) {
                $debt->delete();
            }
            foreach ($budget->eoms as $eoms) {
                $eoms->delete();
            }
            $accounts = $budget->accounts;
            foreach ($accounts as $account) {
                foreach ($account->expenses as $expense) {
                    $expense->delete();
                }
                foreach ($account->transfersFrom as $transferFrom) {
                    $transferFrom->delete();
                }
                foreach ($account->transfersTo as $transferTo) {
                    $transferTo->delete();
                }
            }
            Yii::$app->db->createCommand('SET FOREIGN_KEY_CHECKS=0;')->execute();
            $budget->delete();
            foreach ($accounts as $account) {
                $account->delete();
            }
            Yii::$app->db->createCommand('SET FOREIGN_KEY_CHECKS=1;')->execute();
        }

        if ($reinit) {
            $newBudget = new Budget();
            $newBudget->setAttributes([
                'user_id' => $this->id,
                'name' => 'Default Budget'
            ]);
            $newBudget->save();
            $newBudget->setUpInitialEntities();

            UserMeta::updateUserMeta($this->id, 'active_budget_id', $newBudget->id);
            UserMeta::updateUserMeta($this->id, 'fts_step', '1');
        }
    }


    /**
     * Get the main message conversation of this user.
     * @return Conversation|boolean Returns conversation object on success, false on fail.
     */
    public function getMainConversation()
    {
        if($this->user_role_id == UserRole::SUBCLIENT) {
            $user = $this->userParent->userParent;
        }
        elseif ($this->user_role_id == UserRole::CLIENT) {
            $user = $this;
        }
        else {
            return false;
        }

        $conversation = Conversation::find()
            ->joinWith('conversationsToUsers')
            ->where([
                'and',
                ['name' => 'main'],
                [ConversationToUser::tableName() . '.user_id' => $user->id]
            ])
            ->one();

        // if doesn't exist, create it
        if (!$conversation) {
            $conversation = $this->createConversation('main');
        }
        return $conversation;
    }

    /**
     * Create the main conversation for a user.
     * @param integer $name Name of the conversation.
     * @return Conversation|boolean Returns conversation object on success.
     */
    public function createConversation($name)
    {
        // look for coach
        $coach = $this->getCoach();
        if (!$coach) {
            return false;
        }

        // create conversation
        $conversation = new Conversation();
        $conversation->name = $name;
        $conversation->save();

        // add people to conversation
        $userIds = [];
        $user = ($this->user_role_id == UserRole::CLIENT) ? $this : $this->userParent->userParent;

        $children = $user->userChildren;
        foreach ($children as $c) {
            $userIds[] = $c->user->id;
        }
        $userIds[] = $user->id;
        $userIds[] = $coach->id;

        foreach ($userIds as $u) {
            $ctu = new ConversationToUser();
            $ctu->conversation_id = $conversation->id;
            $ctu->user_id = $u;
            $ctu->save();
        }
        return $conversation;
    }

    /**
     * Get a budget conversatation.
     * @param integer $budgetId ID of the selected mock budget.
     * @return object|boolean Returns conversation object on success, false on fail.
     */
    public function getBudgetConversation($budgetId) {

        if($this->user_role_id == UserRole::SUBCLIENT) {

            $user = $this->userParent->userParent;

        }
        else {

            $user = $this;

        }

        $conversation = Conversation::find()
            ->joinWith('conversationsToUsers')
            ->where([
                'and',
                ['name' => 'budget_' . $budgetId],
                [ConversationToUser::tableName() . '.user_id' => $user->id]
            ])
            ->one();

        return $conversation ? $conversation : false;

    }


    /**
     * Get the clients [User] for this coach;
     * @return array List of users for this coach [clientId => client].
     */
    public function getClients($includeArchived = false)
    {
        if ($this->_clients == null) {
            $clientsQuery = User::find()
                ->select(User::tableName() . '.*, umf.value as first_name, uml.value as last_name')
                ->joinWith('userParent')
                ->leftJoin(UserMeta::tableName() . ' as umf', 'umf.user_id = ' . User::tableName() . '.id AND umf.`key` = \'first_name\'')
                ->leftJoin(UserMeta::tableName() . ' as uml', 'uml.user_id = ' . User::tableName() . '.id AND uml.`key` = \'last_name\'')
                ->where([UserParent::tableName() . '.user_parent_id' => $this->id]);

            if (!$includeArchived) {
                $clientsQuery->andWhere(['archived' => 0]);
            }
            $clients = $clientsQuery
                ->orderBy('last_name desc, first_name desc, username desc')
                ->all();

            $clientsIndexed = [];
            foreach ($clients as $c) {
                $clientsIndexed[$c->id] = $c;
            }
            $this->_clients = $clientsIndexed;
        }
        return $this->_clients;
    }

    public function getConversationClients()
    {
        $clientsQuery = User::find()
            ->alias('u')
            //->select('u.*, umf.value as first_name, uml.value as last_name')
            ->joinWith('userParent')
            //->leftJoin(UserMeta::tableName() . ' as umf', 'umf.user_id = u.id AND umf.`key` = \'first_name\'')
            //->leftJoin(UserMeta::tableName() . ' as uml', 'uml.user_id = u.id AND uml.`key` = \'last_name\'')
            //->leftJoin(ConversationToUser::tableName() . ' as ctu', 'ctu.user_id = u.id')
            //->leftJoin(ConversationMessage::tableName() . ' as cm', 'cm.conversation_id = ctu.conversation_id')
            ->where([UserParent::tableName() . '.user_parent_id' => $this->id]);
            //->orderBy('cm.time DESC, last_name ASC, first_name ASC, username ASC');

        $clientsIndexed = [];
        /** @var User $c */
        foreach ($clientsQuery->all() as $c) {
            $conv = $c->getMainConversation();
            if ($conv) {
                $c->unread_messages = $conv->getUnreadMessagesCount($this->id);
            }
            $clientsIndexed[$c->id] = $c;
        }
        uasort($clientsIndexed, function($a, $b) {
            if ($a->unread_messages < $b->unread_messages) {
                return 1;
            }
            elseif ($a->unread_messages > $b->unread_messages) {
                return -1;
            }
            else {
                if ($a->getFullName() > $b->getFullName()) {
                    return 1;
                }
                else {
                    return -1;
                }
            }
        });

        return $clientsIndexed;
    }

    /**
     * Get a list of conversations related to coach's clients.
     * @param integer $userId ID of the selected user.
     * @return array List of client conversations [clientId => conversation]
     */
    public function getClientConversations($userId)
    {
        $clients = $this->getClients();

        // get all the client conversations
        $clientConversations = [];
        foreach ($clients as $c) {
            $convo = $c->getMainConversation();
            $clientConversations[$c->id] = [
                'conversation' => $convo,
                'unreadMessagesCount' => $convo->getUnreadMessagesCount($userId ? $userId : $this->id)
            ];
        }

        uasort($clientConversations, function($a, $b) {
            return $b['unreadMessagesCount'] - $a['unreadMessagesCount'];
        });

        $clientConversationsSorted = [];
        foreach ($clientConversations as $key => $c) {
            $clientConversationsSorted[$key] = $c['conversation'];
        }

        return $clientConversationsSorted;
    }


    /**
     * Add a notification to a user.
     * @param string $name Name of the notification.
     * @param array $params An array of params to populate the notification (does not include user params).
     * @return object|boolean Returns the notification on success, false on failure.
     */
    public function addNotification($name, $params = [])
    {
        $notificationData = Notification::getNotificationContent($name, $params, $this);
        $subject = $notificationData['subject'];
        $content = $notificationData['content'];

        $notification = new Notification();
        $notification->setAttributes([
            'user_id' => $this->id,
            'notification_template_name' => $name,
            'subject' => $subject,
            'content' => $content
        ]);
        $saved = $notification->save();

        // send email as well if necessary
        if ($this->getMetaValueByKey('email_notifications')) {
            if ($subject && strlen($subject) > 0) {
                $emailSubject = $subject;
            }
            else {
                $emailSubject = 'New notification!';
            }
            DripHelper::sendEmailEvent(
                'notification',
                $this->email,
                $emailSubject,
                [
                    'username' => $this->getFullName(),
                    'notification' => $content,
                    'login_url' => Url::toRoute(['/site/login'], true)
                ]
            );
        }
        return $saved ? $notification : false;
    }

    public function sendNewUserEmail($username, $password)
    {
        DripHelper::sendEmailEvent(
            'new-sub-client',
            $this->email,
            'Confirmation: A user has been added to your account',
            [
                'username' => $username,
                'password' => $password,
                'login_url' => Url::toRoute(['/site/login'], true)
            ]
        );
    }

    /**
     * Initiates the meta value for current step number in FTS
     * @param integer $mainUserId
     * @return integer
     */
    public function initFTSMeta($mainUserId)
    {
        if ($this->id == $mainUserId) {

            $ftsUserMeta = new UserMeta();
            $ftsUserMeta->setAttributes([
                'key'     => 'fts_step',
                'value'   => '1',
                'user_id' => $mainUserId
            ]);
            $ftsUserMeta->save();
            return 1;

        }
        else {

            $ftsUserMeta = UserMeta::findOne(['user_id' => $mainUserId, 'key' => 'fts_step']);
            return $ftsUserMeta->value;

        }
    }

    public function getMainUserId()
    {
        $userParent = UserParent::findOne(['user_id' => $this->id]);
        if ($userParent) {
            if ($userParent->userParent->userRole->name == 'coach') {
                return $userParent->user_id;
            }
            else {
                return $userParent->user_parent_id;
            }
        }
        else {
            return $this->id;
        }
    }

    /**
     * Get the user date format
     * @return string
     */
    public function getDateFormat()
    {
        $dateFormatId = $this->getMetaValueByKey('date_format_id');
        if (!is_numeric($dateFormatId)) {
            $dateFormatId = Yii::$app->params['defaultDateFormatId'];
        }
        return DateFormat::findOne($dateFormatId)->format;
    }

    /**
     * Gets the active budget ID for the user
     * @param integer $mainUserId
     * @return integer
     */
    public function getActiveBudgetId($mainUserId)
    {
        if ($this->getUserRole()->one()->name == 'client') {
            $userMeta = UserMeta::findAllSorted(['user_id' => $this->id]);
            return $userMeta['active_budget_id'];
        }
        else {
            $activeBudgetMeta = UserMeta::findOne(['user_id' => $mainUserId, 'key' => 'active_budget_id']);
            return $activeBudgetMeta->value;
        }
    }

    public function saveAvatar(UploadedFile $avatarObject, $avatarName)
    {
        $filePath = Yii::getAlias('@backend') . '/web/uploads/avatars/' . $avatarName;
        $image = Image::frame($avatarObject->tempName)
            ->thumbnail(new Box(Yii::$app->params['avatarDimensions'][0] + 6, Yii::$app->params['avatarDimensions'][1] + 6), 'outbound')
            ->crop(new Point(3, 3), new Box(Yii::$app->params['avatarDimensions'][0], Yii::$app->params['avatarDimensions'][1]))
            ->save($filePath, ['quality' => 95]);

        $components = Yii::$app->getComponents();
        if (!empty($components['s3'])) {
            Yii::$app->s3->commands()->upload('uploads/avatars/' . $avatarName, $filePath)->execute();
        }
        return $image;
    }

    public function getAvatarUrl()
    {
        $meta = $this->getUserMetaSorted();
        $avatar = isset($meta['avatar']) ? $meta['avatar'] : false;
        $filePath = '/uploads/avatars/' . Formatter::avatarImageName($avatar);
        $components = Yii::$app->getComponents();
        if (empty($components['s3'])) {
            return Url::to('@web' . $filePath);
        }
        else {
            return Yii::$app->assetManager->getHomeUrl() . $filePath;
        }
    }

    /**
     * Return active time zone for the user
     * @return TimeZone
     */
    public function getTimeZone()
    {
        $meta = UserMeta::findAllSorted(['user_id' => $this->id]);
        return TimeZone::findOne((isset($meta['time_zone_id'])) ? $meta['time_zone_id'] : Yii::$app->params['defaultTimeZoneId']);
    }

    /**
     * Saves the login event to the database
     * @param integer $lastActivityTime
     * @param string $userRole
     */
    public function trackLogin($lastActivityTime, $userRole, $userId)
    {
        $login = UserLogin::find()
            ->where(['user_id' => $this->id])
            ->orderBy('id desc')
            ->one();

        if (!$login || !$lastActivityTime || (time() - $lastActivityTime > 7200)) {

            // log login
            $login          = new UserLogin();
            $login->time    = Formatter::datetime(time(), 'UTC', 'Y-n-j', 'php:G:i');
            $login->user_id = $this->id;
            $login->save();

            $this->_consecutiveLoginsNotification($userId);

            // set notifications based on reminders
            if (in_array($userRole, ['client', 'sub-client'])) {

                $this->_remindersToNotifications($userId);

            }

        }
        else {

            $login->duration = strtotime(date('Y-m-d H:i:s')) - strtotime($login->time);
            $login->save();

        }
    }

    protected function _remindersToNotifications($userId) {

        $reminders = NotificationReminder::find()
            ->joinWith('notification')
            ->where([
                'AND',
                ['user_id'  => $userId],
                ['reminded' => 0]
            ])
            ->all();

        foreach ($reminders as $r) {

            $notification = $r->notification;
            $notification->read = 0;
            $notification->archived = 0;
            $saved = $notification->save();
            if ($saved) {
                $r->reminded = 1;
                $r->save();
            }

        }

    }

    protected function _consecutiveLoginsNotification($userId) {

        // check for consecutive logins & add notification
        $loginDates14Days   = [];
        $loginDates21Days   = [];
        $loginDates31Days   = [];
        $todayTime          = strtotime(date('Y-m-d'));
        $time14Days         = strtotime('-13 days', $todayTime);
        $time21Days         = strtotime('-20 days', $todayTime);
        $time31Days         = strtotime('-30 days', $todayTime);
        $loggedIn15DaysAgo  = false;
        $loggedIn22DaysAgo  = false;
        $loggedIn32DaysAgo  = false;
        $date15DaysAgo      = date('Y-m-d', strtotime('-14 days', $todayTime));
        $date22DaysAgo      = date('Y-m-d', strtotime('-21 days', $todayTime));

        $compareDate = date('Y-m-d', strtotime('-32 days'));
        $logins = UserLogin::find()
            ->where([
                'AND',
                ['user_id' => $userId],
                ['>=', 'DATE(time)', $compareDate]
            ])
            ->all();

        foreach ($logins as $l) {

            $time = strtotime($l->time);
            $date = date('Y-m-d', $time);

            if ($time < $time31Days) {

                $loggedIn32DaysAgo = true;

            }

            if ($date == $date15DaysAgo) {

                $loggedIn15DaysAgo = true;

            }

            if ($date == $date22DaysAgo) {

                $loggedIn15DaysAgo = true;

            }

            if ($time > $time14Days) {

                $loginDates14Days[$date] = true;

            }

            if ($time > $time21Days) {

                $loginDates21Days[$date] = true;

            }

            if ($time > $time31Days) {

                $loginDates31Days[$date] = true;

            }

        }

        if ((count(array_keys($loginDates31Days)) == 31) && !$loggedIn32DaysAgo) {
            $notification = Notification::find()
                ->where([
                    'AND',
                    ['user_id' => $userId],
                    ['DATE(created_at)' => date('Y-m-d')],
                    ['notification_template_name' => '31_days_login_streak']
                ])
                ->one();

            if (!$notification) {

                $this->addNotification('31_days_login_streak');

            }

        }
        elseif ((count(array_keys($loginDates21Days)) == 21) && !$loggedIn22DaysAgo) {

            $notification = Notification::find()
                ->where([
                    'AND',
                    ['user_id' => $userId],
                    ['DATE(created_at)' => date('Y-m-d')],
                    ['notification_template_name' => '21_days_login_streak']
                ])
                ->one();

            if (!$notification) {

                $this->addNotification('21_days_login_streak');

            }

        }
        elseif ((count(array_keys($loginDates14Days)) == 14) && !$loggedIn15DaysAgo) {

            $notification = Notification::find()
                ->where([
                    'AND',
                    ['user_id' => $userId],
                    ['DATE(created_at)' => date('Y-m-d')],
                    ['notification_template_name' => '14_days_login_streak']
                ])
                ->one();

            if (!$notification) {

                $this->addNotification('14_days_login_streak');

            }

        }

    }

    /**
     * Performs authentication of the user and returns the result of the authentication
     * @param $username
     * @param $password
     * @param bool $rememberMe
     * @return User|int
     */
    public static function authenticate($username, $password, $rememberMe = true)
    {
        // get user
        $user = User::findOne(['username' => $username, 'archived' => 0]);
        // check if the user is active
        if (!$user) {
            return self::RESULT_INCORRECT;
        }
        if ($user->user_role_id == Yii::$app->params['payingUserRoleId'] && !$user->getUserRegistrations()->filterWhere(['>', 'auth_time', 0])->one()) {
            return self::RESULT_INACTIVE;
        }
        $password = $password . strrev($user->password_salt);
        // check if user found and validate password
        if (!Yii::$app->security->validatePassword($password, $user->password)) {
            return self::RESULT_INCORRECT;
        }

        // check if main user has completed the First Time Setup
        $userParent = UserParent::findOne(['user_id' => $user->id]);
        $mainUserId = ($userParent) ? $userParent->user_parent_id : $user->id;
        $ftsUserMeta = UserMeta::findOne(['user_id' => $mainUserId, 'key' => 'fts_step']);
        if ($userParent) {
            $parentRoleName = $userParent->userParent->userRole->name;
        }
        else {
            $parentRoleName = false;
        }

        if (!$userParent || in_array($parentRoleName, ['coach', 'superadmin']) || ($userParent && $ftsUserMeta && ($ftsUserMeta->value == 'completed'))) {
            if (Yii::$app->user->login($user, $rememberMe ? 3600 * 24 * 30 : 0)){
                // load user meta
                $userMeta = UserMeta::findAllSorted(['user_id' => $user->id]);
                Yii::$app->session->set('userMeta', $userMeta);
                // main account user ID
                if ($userParent && ($parentRoleName == 'client')) {
                    Yii::$app->session->set('mainUserId', $userParent->user_parent_id);
                }
                else {
                    Yii::$app->session->set('mainUserId', $user->id);
                }
                return $user;
            }
            else {
                return self::RESULT_ERROR;
            }
        }
        else {
            return self::RESULT_FTS;
        }
    }

}
