<?php

namespace backend\models\db;

use backend\components\helpers\Formatter;
use backend\models\ChangeMetaModel;
use Yii;

/**
 * This is the model class for table "user_meta".
 *
 * @property integer $id
 * @property integer $user_id
 * @property string $key
 * @property string $value
 * @property integer $visible
 *
 * @property User $user
 */
class UserMeta extends ChangeMetaModel {

    const FIRST_NAME = 'first_name';
    const LAST_NAME = 'last_name';
    const BIRTH_DATE = 'birth_date';
    const MEMBERSHIP_USER = 'membership_user_id';

    /**
     * Special logic to decode all date formats
     */
    const DATE_FIELD_DECODE = 'DATE_FORMAT(' .
        'IF(STR_TO_DATE(value, "%m/%d/%Y"),STR_TO_DATE(value, "%m/%d/%Y"),' .
            'IF(STR_TO_DATE(value, "%d-%m-%Y"),STR_TO_DATE(value, "%d-%m-%Y"),STR_TO_DATE(value, "%Y-%m-%d"))),' .
        'CONCAT(YEAR(CURDATE()), "-%m-%d")' .
    ')';

    const METAKEYS = [
        self::FIRST_NAME => 'First Name',
        self::LAST_NAME => 'Last Name',
        self::BIRTH_DATE => 'Birthday',
        'address_line_1' => 'Address Line 1',
        'address_line_2' => 'Address Line 2',
        'address_line_3' => 'Address Line 3',
        'email_visible' => 'Email Visible',
        'landline_number' => 'Landline Number',
        'cellphone_number' => 'Cellphone Number',
        'skype_name' => 'Skype Account',
        'country_id' => 'Country',
        'avatar' => 'Avatar',
        'dependant' => 'Dependant',
        'date_format_id' => 'Date Format',
        'time_format_id' => 'Time Format',
        'time_zone_id' => 'Time Zone',
        'currency_id' => 'Currency',
        'email_notifications' => 'Email Notifications',
        'currency_format' => 'Currency Format'
    ];

    /**
     * @var string Helper variable for getting the actual birthday date
     */
    public $date;

    /**
     * @inheritdoc
     */
    public static function tableName() {
        return 'user_meta';
    }

    /**
     * @inheritdoc
     */
    public function rules() {
        return [
            [['user_id', 'visible'], 'integer'],
            [['key'], 'string', 'max' => 45],
            [['value'], 'string', 'max' => 200]
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels() {
        return [
            'id' => 'ID',
            'user_id' => 'User ID',
            'key' => 'Key',
            'value' => 'Value',
            'visible' => 'Visible',
        ];
    }

    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        if ($this->scenario == self::SCENARIO_CLONE) {
            return;
        }

        if ($insert) {
            // save change info
            $userMetaChange = new UserMetaChange();
            $userMetaChange->setAttributes([
                'type'         => 'C',
                'user_meta_id' => $this->id,
                'user_id'      => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $userMetaChange->save();
        }
        // user record updated
        else {
            // check if and what changed
            $changes = array();
            $userAttrs = $this->getAttributes();

            foreach ($changedAttributes as $key => $o) {
                if (isset($userAttrs[$key]) && ($userAttrs[$key] != $o)) {
                    $changes[$key] = $o;
                }
            }

            if (count($changes) > 0) {
                $userMetaChange = new UserMetaChange();
                $userMetaChange->setAttributes([
                    'type'         => 'E',
                    'user_meta_id' => $this->id,
                    'user_id'      => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                        Yii::$app->user->identity->id : null
                ]);
                $userMetaChange->save();

                foreach($changes as $key => $value) {
                    $userMetaChangeMeta = new UserMetaChangeMeta();
                    $userMetaChangeMeta->setAttributes([
                        'user_meta_change_id' => $userMetaChange->id,
                        'key'                 => $key,
                        'value'               => (string)$value
                    ]);
                    $userMetaChangeMeta->save();
                }
            }
        }
    }

    public function afterDelete()
    {
        parent::afterDelete();

        $userMetaChange = new UserMetaChange();
        $userMetaChange->setAttributes([
            'type'         => 'D',
            'user_meta_id' => $this->id,
            'user_id'      => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                Yii::$app->user->identity->id : null
        ]);
        $userMetaChange->save();
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUser() {
        return $this->hasOne(User::className(), ['id' => 'user_id']);
    }

    /**
     * Updates existing or creates a new user meta key for the given user
     * @param int $userId
     * @param string $key
     * @param string $value
     * @return bool
     */
    public static function updateUserMeta($userId, $key, $value)
    {
        $userMeta = UserMeta::find()
            ->where([
                'key' => $key,
                'user_id' => $userId
            ])->one();
        if (!$userMeta) {
            $userMeta = new UserMeta();
            $userMeta->setAttributes([
                'key' => $key,
                'user_id' => $userId
            ]);
        }
        $userMeta->value = (string)$value;
        return $userMeta->save();
    }

    public static function findAllSorted($conds)
    {
        $meta = self::findAll($conds);
        $sorted = array();
        foreach ($meta as $m){
            $sorted[$m->key] = $m->value;
        }
        return $sorted;
    }

    public static function getMetaKeys()
    {
        return array_keys(self::METAKEYS);
    }

    public static function getDefaultUserMeta()
    {
        return [
            'time_zone_id'   => Yii::$app->params['defaultTimeZoneId'],
            'time_format_id' => Yii::$app->params['defaultTimeFormatId'],
            'date_format_id' => Yii::$app->params['defaultDateFormatId'],
            'currency_id'    => Yii::$app->params['defaultCurrencyId']
        ];
    }

    public static function getUserDetailsPopulated($userId)
    {
        $firstName   = UserMeta::findOne(['user_id' => $userId, 'key' => 'first_name']);
        $lastName    = UserMeta::findOne(['user_id' => $userId, 'key' => 'last_name']);
        $userAddress = UserMeta::findOne(['user_id' => $userId, 'key' => 'address_line_1']);
        if (!$firstName || empty($firstName->value) ||
            !$lastName || empty($lastName->value) ||
            !$userAddress || empty($userAddress->value)) {
            return true;
        } else {
            return false;
        }
    }

    public static function getUsersBirthdays(array $userParents, $onlyIn30Days = false)
    {
        $query = UserMeta::find()
            ->select([
                UserMeta::tableName() . '.*',
                self::DATE_FIELD_DECODE . ' AS date'
            ])
            ->joinWith('user.userParent')
            ->with('user')
            ->where([
                'AND',
                ['key' => 'birth_date'],
                ['!=', 'value', ''],
                ['user_parent_id' => $userParents]
            ]);
        // birthdays in 30 days
        if ($onlyIn30Days) {
            $query->andWhere([
                self::DATE_FIELD_DECODE => Formatter::localDatetimeToUtcDatetime(strtotime('+30 days'), 'Y-m-d')
            ]);
        }
        else {
            $query->andWhere([
                'BETWEEN', self::DATE_FIELD_DECODE, Formatter::localDatetimeToUtcDatetime(time(), 'Y-m-d'), Formatter::localDatetimeToUtcDatetime(strtotime('+1 MONTH'), 'Y-m-d')
            ]);
        }

        return $query->all();
    }


    public function getKeyLabel()
    {
        $label = @(self::METAKEYS[$this->key]);
        if (empty($label)) {
            return $this->key;
        }
        return $label;
    }

}
