<?php

namespace backend\models\db;

use backend\components\helpers\DripHelper;
use backend\components\helpers\Formatter;
use backend\components\helpers\Stripe;
use Yii;
use yii\helpers\Url;

/**
 * This is the model class for table "user_plans".
 *
 * @property integer $id
 * @property integer $user_id
 * @property integer $user_plan_type_id
 * @property integer $user_plan_variation_id
 * @property string $stripe_subscription_id
 * @property string $stripe_charge_id
 * @property string $stripe_coupon_id
 * @property string $start_time
 * @property string $valid_until
 * @property integer $is_cancelled
 *
 * @property User $user
 * @property UserPlanType $userPlanType
 * @property UserPlanVariation $userPlanVariation
 * @property InvoiceItem $invoiceItems
 */
class UserPlan extends \yii\db\ActiveRecord
{
    const INVOICE = 1;
    const CHANGE_PLAN = 2;
    const EXPIRED = 3;

    const EXPIRY_NOTIFICATION_DAYS = 7;

    const BOOTCAMP_FREE_DAYS = 40;


    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'user_plans';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['user_id', 'user_plan_type_id', 'user_plan_variation_id', 'is_cancelled'], 'integer'],
            [['start_time', 'valid_until', 'stripe_subscription_id', 'stripe_coupon_id'], 'safe']
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'user_id' => 'User',
            'user_plan_type_id' => 'User Plan Type',
            'start_time' => 'Start Time',
            'is_cancelled' => 'Is Cancelled'
        ];
    }

    /**
     * @return int Unix timestamp for expiration time
     */
    public function getExpirationTime()
    {
        return strtotime(
            date(
                'Y-m-d 23:59:59',
                strtotime($this->valid_until ? $this->valid_until : $this->start_time)
            )
        );
    }

    /**
     * @return string Return the expired
     */
    public function getFormattedExpirationTime()
    {
        return Yii::$app->formatter->asDatetime($this->getExpirationTime());
    }

    /**
     * @return string Return the date expired
     */
    public function getFormattedExpirationDate()
    {
        return Yii::$app->formatter->asDate($this->getExpirationTime());
    }

    /**
     * @return string Return the date started
     */
    public function getFormattedStartDate()
    {
        return Yii::$app->formatter->asDate($this->start_time);
    }

    /**
     * @return string Return the days passed of plan
     */
    public function getDaysPassed($param_date_finish = null)
    {
        if (is_null($param_date_finish)) {
            $date_finish = "now";
        } else {
            $date_finish = $param_date_finish;
        }

        return date_diff(date_create($this->start_time), date_create($date_finish))->days;
    }

    /**
     * @return string Return the days left of the plan
     */
    public function getDaysLeft($param_date_finish = null)
    {
        if (is_null($param_date_finish)) {
            $date_finish = "now";
        } else {
            $date_finish = $param_date_finish;
        }

        $end = date_create(date('Y-m-d', $this->getExpirationTime()));
        $now = date_create($date_finish);
        return $end <= $now ?
            0 :
            date_diff($end, $now)->days + 1;
    }

    /**
     * @return string Return the percentage completed of plan
     */
    public function getPercentageCompleted()
    {
        $daysPassed = $this->getDaysPassed(date('Y-m-d', $this->getExpirationTime()));
        return $daysPassed > 0 ?
            (int)(($this->getDaysPassed() * 100) / $daysPassed) :
            0;
    }

    /**
     * @return string Message based on expiration time (Active, Expiring soon, Expired)
     */
    public function getStatus()
    {
        $expTime = $this->getExpirationTime();
        $now = time();
        if ($expTime < $now) {
            return 'Expired';
        }

        $week = strtotime('+2 week'); // Number of weeks for "expiring soon"
        if ($expTime < $week) {
            return 'Expiring soon!';
        }

        return 'Active';
    }

    /**
     * @return bool Whether the user plan is expired or not
     */
    public function getIsExpired()
    {
        return !empty($this->userPlanVariation->plan_length) && // users without the end time can't expire
            !empty(floatval($this->userPlanVariation->price)) && // free plans can't expire
            $this->getExpirationTime() < time();
    }

    public function getNextStartTime()
    {
        return $this->getExpirationTime();
    }

    public function getNextExpirationTime()
    {

        return strtotime(
            UserPlanVariation::getDaysMonths($this->userPlanVariation->plan_length, $this->userPlanVariation->plan_length_in_weeks),
            $this->getNextStartTime()
        );
    }

    public function getFormattedNextStartDate()
    {
        return Yii::$app->formatter->asDate($this->getNextStartTime());
    }

    public function getFormattedNextExpirationDate()
    {
        return Yii::$app->formatter->asDate($this->getNextExpirationTime());
    }

    /**
     * Get the latest user plan (not necessarily an active one)
     * @param integer $userId
     * @return null|UserPlan
     */
    public static function getLatestPlan($userId = null)
    {
        return self::getActivePlan($userId);
    }

    /**
     * Gets the current active plan
     * @param integer $userId
     * @return null|UserPlan
     */
    public static function getActivePlan($userId = null)
    {
        if (!$userId) {
            $userId = Yii::$app->user->identity->id;
        }

        $withExpiry = UserPlan::find()
            ->where(['user_id' => $userId])
            ->andWhere(['<=', 'start_time', Formatter::localDatetimeToUtcDatetime(time(), 'Y-m-d H:i:s')])
            ->andWhere(['not', ['valid_until' => null]])
            ->orderBy([
                'DATE(valid_until)' => SORT_DESC,
                'DATE(start_time)' => SORT_ASC,
                'id' => SORT_DESC
            ])
            ->limit(1)
            ->one();
        $withNoExpiry = UserPlan::find()
            ->where(['user_id' => $userId])
            ->andWhere(['<=', 'start_time', Formatter::localDatetimeToUtcDatetime(time(), 'Y-m-d H:i:s')])
            ->andWhere(['valid_until' => null])
            ->andWhere(['is_cancelled' => 0])
            ->orderBy([
                'DATE(start_time)' => SORT_DESC,
                'id' => SORT_DESC
            ])
            ->limit(1)
            ->one();
        if ($withNoExpiry) {
            if ($withExpiry && $withExpiry->valid_until >= Formatter::localDatetimeToUtcDatetime(time(), 'Y-m-d H:i:s')) {
                return $withExpiry;
            }
            return $withNoExpiry;
        }
        return $withExpiry;
    }

    public function getNextAction($daysBefore = null)
    {
        // simply get the next action, regardless of the time left
        if ($daysBefore == null) {
            $daysBefore = $this->getDaysLeft();
        }
        if ($this->is_cancelled) {
            return false;
        }
        if ($this->getDaysLeft() != $daysBefore) {
            return false;
        }
        // notify only every $plan_length months
        $endDate = date('Y-m-d', strtotime($this->valid_until));
        if ($endDate < date('Y-m-d')) {
            return self::EXPIRED;
        }
        if ($endDate != date('Y-m-d')) {
            return false;
        }
        return false;
    }

    public function getLatestInvoice($paid = null)
    {
        $where = ['user_id' => $this->user_id];
        if ($paid !== null) {
            $where['is_paid'] = $paid ? 1 : 0;
        }
        return Invoice::find()
            ->where($where)
            ->orderBy('id DESC')
            ->one();
    }

    /**
     * Creates a copy of the current plan so the next billing period can be paid for
     * @return UserPlan
     */
    public function cloneForNewPeriod()
    {
        $plan = new UserPlan();
        $plan->setAttributes([
            'user_id'                => $this->user_id,
            'user_plan_type_id'      => $this->user_plan_type_id,
            'user_plan_variation_id' => $this->user_plan_variation_id,
            'start_time'             => Formatter::localDatetimeToUtcDatetime($this->getExpirationTime(), 'Y-m-d 00:00:00')
        ]);
        $plan->save();

        return $plan;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUser()
    {
        return $this->hasOne(User::className(), ['id' => 'user_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserPlanType()
    {
        return $this->hasOne(UserPlanType::className(), ['id' => 'user_plan_type_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserPlanVariation()
    {
        return $this->hasOne(UserPlanVariation::className(), ['id' => 'user_plan_variation_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    protected function getCoachId()
    {
        return $this->hasOne(UserMeta::className(),
            ['user_id' => 'id'])->andWhere(['key' => UserMeta::COACH])->via('user');
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getCoach()
    {
        return $this->hasOne(User::className(), ['id' => 'value'])->via('coachId');
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getInvoiceItems()
    {
        return $this->hasMany(InvoiceItem::className(), ['user_plan_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserPlanTypeDuration()
    {
        return $this->hasOne(UserPlanTypeDuration::className(), ['user_plan_type_id' => 'id'])
            ->viaTable(UserPlanType::tableName(), ['id' => 'user_plan_type_id']);
    }

    /**
     * @inheritdoc
     */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);

        // create default budget
        if ($insert) {

            if (!Budget::find()->where(['user_id' => $this->user_id])->exists()) {
                // create the default user budget
                $this->user->addNewBudget('Default Budget');
            }

            $userMeta = UserMeta::findAllSorted(['user_id' => $this->user_id]);
            if (empty($userMeta['active_budget_id'])) {
                // set active budget in meta
                $budget = Budget::findOne(['user_id' => $this->user_id]);
                $budgetUserMeta = new UserMeta();
                $budgetUserMeta->setAttributes([
                    'user_id' => $this->user_id,
                    'key' => 'active_budget_id',
                    'value' => (string)$budget->id
                ]);
                $budgetUserMeta->save();

                // update the user meta data
                $userMeta['active_budget_id'] = (string)$budget->id;
                Yii::$app->session->set('userMeta', $userMeta);
            }

        }
    }

    public function sendUpcomingEmail($amount)
    {
        // only for yearly subscriptions
        if ($this->userPlanVariation->plan_length_in_weeks || $this->userPlanVariation->plan_length != 12 ||
            $amount <= 0) {
            return;
        }
        $user = $this->user;

        DripHelper::sendEmailEvent(
            'invoice-upcoming',
            $user->email,
            'Upcoming Invoice from Grandma’s Jars',
            [
                'username'          => $user->getFullName(),
                'plan_name'         => $this->userPlanVariation->name,
                'amount'            => number_format($amount / 100, 2),
                'subscriptions_url' => Url::toRoute(['/subscription/index'], true)
            ]
        );
    }


    /**
     * @param UserPlan|UserPlanVariation $planOrVariation
     * @param integer $userId
     * @param string $tokenId
     * @param bool $withKickstart
     * @return bool
     */
    public static function createAndSubscribe($planOrVariation, $userId, $tokenId, $withKickstart = true)
    {
        if (get_class($planOrVariation) == 'backend\models\db\UserPlanVariation') {
            $userPlan = new UserPlan();
            $userPlan->setAttributes([
                'user_id' => $userId,
                'user_plan_type_id' => $planOrVariation->user_plan_type_id,
                'user_plan_variation_id' => $planOrVariation->id,
                'start_time' => Formatter::localDatetimeToUtcDatetime(time(), 'Y-m-d 00:00:00')
            ]);
            if ($withKickstart) {
                // give some kickstart time before the invoice is paid
                $userPlan->valid_until = Formatter::localDatetimeToUtcDatetime(strtotime('+1 day'), 'Y-m-d 23:59:59');
            }
            if (!$userPlan->save()) {
                return false;
            }
        }
        else {
            $userPlan = $planOrVariation;
            $userPlan->start_time = Formatter::localDatetimeToUtcDatetime(time(), 'Y-m-d 00:00:00');
            if ($withKickstart) {
                // give some kickstart time before the invoice is paid
                $userPlan->valid_until = Formatter::localDatetimeToUtcDatetime(strtotime('+1 day'), 'Y-m-d 23:59:59');
            }
            if (!$userPlan->save()) {
                return false;
            }
        }

        $user = User::findOne($userId);
        $stripe = new Stripe();
        $customerId = $stripe->saveCustomer($user, $tokenId);
        if (!$customerId) {
            return false;
        }
        if ($stripe->subscribeCustomerToPlan($customerId, $userPlan) !== null) {
            return $customerId;
        }
        return null;
    }

    /**
     * @param UserPlan $userPlan
     * @param integer $userId
     * @param string $tokenId
     * @param bool $withKickstart
     * @return bool
     */
    public static function createAndCharge(UserPlan $userPlan, $userId, $tokenId, $withKickstart = true)
    {
        $userPlan->start_time = Formatter::localDatetimeToUtcDatetime(time(), 'Y-m-d 00:00:00');
        if ($withKickstart) {
            // give some kickstart time before the invoice is paid
            $userPlan->valid_until = Formatter::localDatetimeToUtcDatetime(strtotime('+1 day'), 'Y-m-d 23:59:59');
        }
        if (!$userPlan->save()) {
            return false;
        }

        $user = User::findOne($userId);
        $stripe = new Stripe();
        $customerId = $stripe->saveCustomer($user, $tokenId);
        if (!$customerId) {
            return false;
        }
        return $stripe->chargeCustomer($customerId, $userPlan) !== null;
    }

    /**
     * @param UserPlan $userPlan
     * @param string $customerId
     * @param string $amount
     * @return bool
     */
    public static function chargeOnly(UserPlan $userPlan, $customerId, $amount)
    {
        $stripe = new Stripe();
        return $stripe->chargeCustomer($customerId, $userPlan, $amount) !== null;
    }

    /**
     * @param $stripeChargeId
     * @return UserPlan
     */
    public static function getByStripeChargeID($stripeChargeId)
    {
        return UserPlan::find()
            ->where([
                'stripe_charge_id' => $stripeChargeId
            ])
            ->orderBy([
                'DATE(valid_until)' => SORT_DESC,
                'DATE(start_time)' => SORT_ASC,
                'id' => SORT_DESC
            ])
            ->one();
    }

    /**
     * @param string $stripeSubscriptionId
     * @return UserPlan
     */
    public static function getByStripeSubscriptionID($stripeSubscriptionId)
    {
        return UserPlan::find()
            ->where([
                'stripe_subscription_id' => $stripeSubscriptionId
            ])
            ->orderBy([
                'DATE(valid_until)' => SORT_DESC,
                'DATE(start_time)' => SORT_ASC,
                'id' => SORT_DESC
            ])
            ->one();
    }

    /**
     * Gets an unassociated user plan record for the given stripe customer ID
     * (and automatically associates the user plan with Stripe subscription)
     * @param string $stripeCustomerId
     * @param array $lines
     * @param bool|string $subscriptionId
     * @return UserPlan|bool
     */
    public static function getByStripeCustomerID($stripeCustomerId, $lines = array(), $subscriptionId = false)
    {
        $userMeta = UserMeta::find()
            ->where([
                'key' => 'stripe_customer_id',
                'value' => $stripeCustomerId
            ])
            ->one();
        if (!$userMeta) {
            // trying to find by email
            $stripe = new Stripe();
            $customer = $stripe->getCustomer($stripeCustomerId);
            $user = User::findOne(['email' => $customer->email]);
            if ($user) {
                $userMeta = new UserMeta();
                $userMeta->setAttributes([
                    'user_id' => $user->id,
                    'key' => 'stripe_customer_id',
                    'value' => $stripeCustomerId
                ]);
                $userMeta->save();
            }
            else {
                return false;
            }
        }
        foreach ($lines as $line) {
            $userPlan = UserPlan::find()
                ->joinWith('userPlanVariation')
                ->where([
                    UserPlan::tableName() . '.user_id' => $userMeta->user_id,
                    UserPlan::tableName() . '.stripe_subscription_id' => null,
                    UserPlanVariation::tableName() . '.stripe_plan_id' => $line['plan']['id']
                ])
                ->orderBy([
                    'DATE(valid_until)' => SORT_DESC,
                    'DATE(start_time)' => SORT_ASC,
                    UserPlan::tableName() . '.id' => SORT_DESC
                ])
                ->one();
            if ($userPlan) {
                $userPlan->updateAttributes([
                    'stripe_subscription_id' => $line['id']
                ]);
                return $userPlan;
            }
            // automatically create the user plan if not found
            elseif ($subscriptionId) {
                $variation = UserPlanVariation::findOne(['stripe_plan_id' => $line['plan']['id']]);
                if ($variation) {
                    $userPlan = new UserPlan();
                    $userPlan->setAttributes([
                        'user_id' => $userMeta->user_id,
                        'user_plan_type_id' => $variation->user_plan_type_id,
                        'user_plan_variation_id' => $variation->id,
                        'stripe_subscription_id' => $subscriptionId,
                        'start_time' => Formatter::localDatetimeToUtcDatetime(time(), 'Y-m-d 00:00:00'),
                        // give some kickstart time before the invoice is paid
                        'valid_until' => Formatter::localDatetimeToUtcDatetime(strtotime('+1 day'), 'Y-m-d 23:59:59')
                    ]);
                    if ($userPlan->save()) {
                        return $userPlan;
                    }
                }
            }
        }
        return false;
    }

    public function updatePlanFromStripe($eventData)
    {
        if (count($eventData['items']['data']) > 0) {
            $line = array_shift($eventData['items']['data']);
            $variation = UserPlanVariation::findOne(['stripe_plan_id' => $line['plan']['id']]);
            if ($variation) {
                $this->updateAttributes([
                    'user_plan_type_id' => $variation->user_plan_type_id,
                    'user_plan_variation_id' => $variation->id,
                    'valid_until' => date('Y-m-d 23:59:59', strtotime('+' . Stripe::GRACE_PERIOD_DAYS . ' days', $eventData['current_period_end']))
                ]);
                return true;
            }
        }
        return false;
    }

    public function toMembershipLevel()
    {
        if (stripos($this->userPlanType->name, 'conquer') !== false) {
            return '6';
        }
        elseif (stripos($this->userPlanType->name, 'bootcamp') === false &&
            stripos($this->userPlanType->name, 'free') === false &&
            stripos($this->userPlanType->name, 'application') === false) {
            return '1,2';
        }
        elseif (stripos($this->userPlanType->name, 'bootcamp access') !== false) {
            return '5';
        }
        else {
            return '1';
        }
    }

}

