<?php

namespace backend\models\db;


/**
 * This is the model class for table "user_plan_types".
 *
 * @property integer $id
 * @property string $name
 * @property string $description
 * @property integer $consultation_amount
 * @property string $consultation_period
 *
 * @property UserPlanDetail[] $userPlanDetails
 * @property UserPlanTypeDuration[] $userPlanTypeDuration
 * @property UserPlan[] $userPlans
 */
class UserPlanType extends \yii\db\ActiveRecord
{
    const PLAN_TYPE_BOOTCAMP = 'bootcamp';
    const PLAN_TYPE_BOOTCAMP_ACCESS = 'bootcamp-access';
    const PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC = 'discount-conquer-your-cc';
    const PLAN_TYPE_CONQUER_YOUR_CC = 'conquer-your-cc';
    const PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_1_YEAR = 'discount-conquer-your-cc-one-year';
    const PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_3_MONTHS = 'discount-conquer-your-cc-three-months';
    const PLAN_TYPE_CONQUER_YOUR_CC_1_YEAR = 'conquer-your-cc-one-year';
    const PLAN_TYPE_CONQUER_YOUR_CC_3_MONTHS = 'conquer-your-cc-three-months';
    const PLAN_TYPE_FREE = 'free';
    const PLAN_TYPE_FREE_12_MONTHS = 'free-12-months';
    const PLAN_TYPE_APP = 'app';
    const PLAN_TYPE_MEMBERSHIP = 'membership';
    const PLAN_TYPE_MAINTENANCE = 'maintenance';
    const PLAN_TYPE_MAINTENANCE_OLD = 'maintenance-old';
    const PLAN_TYPE_COACHING_MAINTENANCE = 'coaching-maintenance';
    const PLAN_TYPE_COACHING = 'one-on-one-coaching';
    const PLAN_TYPE_FINANCIAL_COACHING_PROGRAM = 'financial-coaching-program';
    const PLAN_TYPE_MASTERING_MONEY_COACHING_PROGRAM = 'mastering-money-coaching-program';
    const PLAN_TYPE_KICKSTART = 'kickstart';
    const PLAN_TYPE_FINANCIAL_KICKSTARTER = 'financial-kickstarter';
    const PLAN_TYPE_MOSAIC = 'mosaic';
    const PLAN_TYPE_STRONG_MONEY_SKILLS = 'strong-money-skills';

    const GRAVITY_97 = 'gravity-9700'; // yearly app plan


    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'user_plan_types';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            ['consultation_amount', 'integer'],
            ['name', 'string', 'max' => 200],
            ['consultation_period', 'string', 'max' => 20]
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'name' => 'Name',
            'consultation_amount' => 'Consultation Amount',
            'consultation_period' => 'Consultation Period',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserPlanDetails()
    {
        return $this->hasMany(UserPlanDetail::className(), ['user_plan_type_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserPlanTypeDuration()
    {
        return $this->hasOne(UserPlanTypeDuration::className(), ['user_plan_type_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserPlans()
    {
        return $this->hasMany(UserPlan::className(), ['user_plan_type_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserPlanVariations()
    {
        return $this->hasMany(UserPlanVariation::className(), ['user_plan_type_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserMeta($key)
    {
        return $this->hasMany(UserMeta::className(), ['user_id' => 'user_id'])
            ->andWhere(['key' => $key])->via('userPlans');
    }

    protected function getCoachesId()
    {
        return $this->getUserMeta(UserMeta::COACH);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getCoaches()
    {
        return $this->hasMany(User::className(), ['id' => 'value'])->via('coachesId');
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getActiveUserPlans()
    {
        return $this->hasMany(UserPlan::className(), ['user_plan_type_id' => 'id'])
            ->andWhere('NOW()<=DATE_ADD(start_time, INTERVAL consultation_period MONTH)');
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getActiveUserMeta($key)
    {
        return $this->hasMany(UserMeta::className(), ['user_id' => 'user_id'])
            ->andWhere(['key' => $key])->via('activeUserPlans');
    }

    protected function getActiveCoachesId()
    {
        return $this->getUserMeta(UserMeta::COACH)
            ;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getActiveCoaches()
    {
        return $this->hasMany(User::className(), ['id' => 'value'])->via('activeCoachesId');
    }

    /**
     * Gets the first variation for the type
     * @return UserPlanVariation
     */
    public function getDefaultPlanVariation()
    {
        return UserPlanVariation::find()
            ->where([
                'user_plan_type_id' => $this->id
            ])
            ->orderBy('name ASC')
            ->one();
    }

    public function getVariationNames()
    {
        $variations = [];
        foreach ($this->getUserPlanVariations()->all() as $variation) {
            $variations[] = $variation->getCleanName();
        }
        return $variations;
    }

    /**
     * Get the plan type of the current plan
     * @return string
     */
    public function getPlanType()
    {
        if (stripos($this->name, '(discount) conquer your credit card + application 1 year') === 0) {
            return self::PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_1_YEAR;
        }
        if (stripos($this->name, '(discount) conquer your credit card + application 3 months') === 0) {
            return self::PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_3_MONTHS;
        }
        if (stripos($this->name, 'conquer your credit card + application 1 year') === 0) {
            return self::PLAN_TYPE_CONQUER_YOUR_CC_1_YEAR;
        }
        if (stripos($this->name, 'conquer your credit card + application 3 months') === 0) {
            return self::PLAN_TYPE_CONQUER_YOUR_CC_3_MONTHS;
        }
        if (stripos($this->name, 'free 12 months') !== false) {
            return self::PLAN_TYPE_FREE_12_MONTHS;
        }
        if (stripos($this->name, 'free') !== false) {
            return self::PLAN_TYPE_FREE;
        }
        if (stripos($this->name, 'application') !== false) {
            return self::PLAN_TYPE_APP;
        }
        if (stripos($this->name, 'membership') !== false) {
            return self::PLAN_TYPE_MEMBERSHIP;
        }
        if (stripos($this->name, 'coaching maintenance') !== false) {
            return self::PLAN_TYPE_COACHING_MAINTENANCE;
        }
        if (stripos($this->name, 'maintenance monthly') !== false) {
            return self::PLAN_TYPE_MAINTENANCE_OLD;
        }
        if (stripos($this->name, 'maintenance') !== false) {
            return self::PLAN_TYPE_MAINTENANCE;
        }
        if (stripos($this->name, 'one on one coaching') !== false) {
            return self::PLAN_TYPE_COACHING;
        }
        if (stripos($this->name, 'financial coaching program') !== false) {
            return self::PLAN_TYPE_FINANCIAL_COACHING_PROGRAM;
        }
        if (stripos($this->name, 'financial kickstarter') !== false) {
            return self::PLAN_TYPE_FINANCIAL_KICKSTARTER;
        }
        if (stripos($this->name, 'kickstart') !== false) {
            return self::PLAN_TYPE_KICKSTART;
        }
        if (stripos($this->name, 'mosaic') !== false) {
            return self::PLAN_TYPE_MOSAIC;
        }
        if (stripos($this->name, 'strong money skills') !== false) {
            return self::PLAN_TYPE_STRONG_MONEY_SKILLS;
        }
        if (stripos($this->name, 'bootcamp access') !== false) {
            return self::PLAN_TYPE_BOOTCAMP_ACCESS;
        }

        return self::PLAN_TYPE_BOOTCAMP;
    }

    /**
     * Gets the UserPlanType object for the given plan type option
     * @param int $planType
     * @return null|UserPlanType
     */
    public static function getPlanTypeObject($planType)
    {
        switch ($planType) {
            case self::PLAN_TYPE_BOOTCAMP:
                return UserPlanType::find()->where(['name' => 'bootcamp'])->one();
            case self::PLAN_TYPE_BOOTCAMP_ACCESS:
                return UserPlanType::find()->where(['like', 'name', 'bootcamp access'])->one();
            case self::PLAN_TYPE_FREE_12_MONTHS:
                return UserPlanType::find()->where(['name' => 'free 12 months'])->one();
            case self::PLAN_TYPE_FREE:
                return UserPlanType::find()->where(['name' => 'free'])->one();
            case self::PLAN_TYPE_APP:
            case self::GRAVITY_97:
                return UserPlanType::find()->where(['like', 'name', 'application'])->one();
            case self::PLAN_TYPE_MEMBERSHIP:
                return UserPlanType::find()->where(['like', 'name', 'membership'])->one();
            case self::PLAN_TYPE_MAINTENANCE_OLD:
                return UserPlanType::find()->where(['name' => 'maintenance monthly'])->one();
            case self::PLAN_TYPE_MAINTENANCE:
                return UserPlanType::find()->where(['name' => 'maintenance'])->one();
            case self::PLAN_TYPE_COACHING_MAINTENANCE:
                return UserPlanType::find()->where(['like', 'name', 'coaching maintenance'])->one();
            case self::PLAN_TYPE_COACHING:
                return UserPlanType::find()->where(['like', 'name', 'one on one coaching'])->one();
            case self::PLAN_TYPE_FINANCIAL_COACHING_PROGRAM:
                return UserPlanType::find()->where(['like', 'name', 'financial coaching program'])->one();
            case self::PLAN_TYPE_MASTERING_MONEY_COACHING_PROGRAM:
                return UserPlanType::find()->where(['like', 'name', 'mastering money coaching program'])->one();
            case self::PLAN_TYPE_KICKSTART:
                return UserPlanType::find()->where(['like', 'name', 'kickstart'])->where(['not like', 'name', 'financial kickstarter'])->one();
            case self::PLAN_TYPE_FINANCIAL_KICKSTARTER:
                return UserPlanType::find()->where(['like', 'name', 'financial kickstarter'])->one();
            case self::PLAN_TYPE_MOSAIC:
                return UserPlanType::find()->where(['like', 'name', 'mosaic'])->one();
            case self::PLAN_TYPE_STRONG_MONEY_SKILLS:
                return UserPlanType::find()->where(['like', 'name', 'strong money skills'])->one();
            case self::PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_1_YEAR:
                return UserPlanType::find()->where(['name' => '(discount) conquer your credit card + application 1 year'])->one();
            case self::PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_3_MONTHS:
                return UserPlanType::find()->where(['name' => '(discount) conquer your credit card + application 3 months'])->one();
            case self::PLAN_TYPE_CONQUER_YOUR_CC_1_YEAR:
                return UserPlanType::find()->where(['name' => 'conquer your credit card + application 1 year'])->one();
            case self::PLAN_TYPE_CONQUER_YOUR_CC_3_MONTHS:
                return UserPlanType::find()->where(['name' => 'conquer your credit card + application 3 months'])->one();
            default:
                return null;
        }
    }

    public static function getPlanVariationObject($planType)
    {
        if (strpos($planType, 'gravity') === 0) {
            return UserPlanVariation::find()->where(['like', 'name', 'application yearly'])->one();
        }

        switch ($planType) {
            case self::GRAVITY_97:
                return UserPlanVariation::find()->where(['like', 'name', 'application yearly'])->one();
            default:
                return null;
        }
    }

}
