<?php

namespace backend\models\db;

use Yii;

/**
 * This is the model class for table "user_plan_variations".
 *
 * @property integer $id
 * @property string $name
 * @property string $stripe_plan_id
 * @property integer $user_plan_type_id
 * @property integer $plan_length
 * @property integer $plan_length_in_weeks
 * @property integer $access_length
 * @property integer $cancel_after
 * @property string $price
 * @property string $upfront_payment
 *
 * @property UserPlanType $userPlanType
 */
class UserPlanVariation extends \yii\db\ActiveRecord
{
    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'user_plan_variations';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['name', 'stripe_plan_id', 'user_plan_type_id', 'plan_length', 'price'], 'required'],
            [['user_plan_type_id', 'plan_length', 'plan_length_in_weeks'], 'integer'],
            [['price', 'upfront_payment'], 'number'],
            [['name', 'stripe_plan_id'], 'string', 'max' => 255],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'name' => 'Name',
            'stripe_plan_id' => 'Stripe Plan ID',
            'user_plan_type_id' => 'User Plan Type ID',
            'plan_length' => 'Plan Length',
            'price' => 'Price',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserPlanType()
    {
        return $this->hasOne(UserPlanType::className(), ['id' => 'user_plan_type_id']);
    }


    public function getCleanName()
    {
        $parentName = $this->getUserPlanType()->one()->name;
        $pos = stripos($this->name, $parentName);
        if ($pos !== false) {
            $name = trim(substr($this->name, $pos + strlen($parentName)));
        }
        else {
            $name = $this->name;
        }
        if (empty($name)) {
            $period = $this->getPricePeriod('');
            return empty($period) ? $period : ucfirst($period . 'ly');
        }
        return $name;
    }

    /**
     * @return string Return the price per month of plan
     */
    public function getPricePerMonth()
    {
        if ($this->plan_length_in_weeks) {
            return round($this->plan_length > 0 ?
                $this->price * 4.35 / $this->plan_length :
                $this->price * 4.35, 2);
        }
        else {
            return round($this->plan_length > 0 ?
                $this->price / $this->plan_length :
                $this->price, 2);
        }
    }

    public function getPrice()
    {
        return round($this->price, 2);
    }

    public function getPriceText($full = false, $withGst = true)
    {
        $price = $this->getPrice();
        if ($full) {
            return $price ? '$' . $price . $this->getPricePeriod('/') . ($withGst ? ' + GST' : '') : 'FREE';
        }
        else {
            return $price ? '$' . $price . ($withGst ? ' + GST' : '') : 'FREE';
        }
    }

    public function getPricePeriod($prefix = 'per ')
    {
        if ($this->plan_length_in_weeks) {
            return $prefix . 'week';
        }
        $period = '';
        switch ($this->plan_length) {
            case 1:
                $period = $prefix . 'month';
                break;
            case 3:
                $period = $prefix . 'quarter';
                break;
            case 12:
                $period = $prefix . 'year';
                break;
        }
        return $period;
    }

    public function getBillingMonths()
    {
        if ($this->cancel_after) {
            return $this->cancel_after - ($this->upfront_payment ? 1 : 0);
        }
        return null;
    }

    public static function getDaysMonths($count, $inWeeks = false)
    {
        if ($inWeeks) {
            return '+' . $count . ' weeks';
        }
        else {
            return '+' . $count . ' ' . ($count > 12 ? 'days' : 'months');
        }
    }
}
