<?php

namespace backend\models\form;

use backend\components\validators\Currency;
use backend\models\form\common\DateField;
use backend\models\form\common\DateNextDueField;
use backend\models\form\common\InterestFreeEndDateField;
use backend\models\form\common\NextDueDateField;
use Yii;

use backend\models\db\Account;
use backend\models\db\Debt;

use backend\components\helpers\Formatter;
use backend\components\validators\DateStart;

/**
 * New jar form
 */
class AddDebt extends \yii\base\Model
{
    use DateField, NextDueDateField, InterestFreeEndDateField;

    public $name;
    public $amount;
    public $account_id;
    public $payment;
    public $debt_type_id;
    public $is_debt_active = 0;
    public $interest = 0;
    public $frequency;
    public $date;
    public $next_due_date;
    public $has_interest_free_period;
    public $interest_free_end_date;
    public $tax_deductible = 0;
    public $budget_id;
    public $debt_id = -1;


    public function rules() {

        // validate input
        $validationRules = [
            ['name', 'required'],
            ['name', 'string', 'min' => 3],
            ['name', 'debtNameValid'],
            ['amount', 'required'],
            ['amount', Currency::className(), 'min' => 0],
            ['account_id', 'required', 'message' => 'Please select an account.'],
            ['account_id', 'integer'],
            ['account_id', 'accountIdValid'],
            ['interest', 'number', 'min' => 0],
            ['date', 'required'],
            ['date', DateStart::className(), 'fieldName' => 'debt', 'on' => 'create'],
            ['date', 'date', 'format' => Formatter::phpDateCodeToDateCode(Yii::$app->session->get('dateFormat')), 'message' => 'Incorrect date format.'],
            ['next_due_date', 'safe'],
            ['next_due_date', 'date', 'min' => time(), 'tooSmall' => 'Next Due Date must be in the future.', 'format' => Formatter::phpDateCodeToDateCode(Yii::$app->session->get('dateFormat')), 'message' => 'Incorrect date format.'],
            ['payment', 'required'],
            ['payment', Currency::className(), 'min' => 0],
            ['debt_type_id', 'required'],
            ['debt_type_id', 'integer'],
            ['interest_free_end_date', 'required', 'when' => function($model) {
                return $model->has_interest_free_period;
            }, 'whenClient' => "function (attribute, value) {
                return $('#adddebt-has_interest_free_period').is(':checked');
            }"],
            ['interest_free_end_date', 'date', 'format' => Formatter::phpDateCodeToDateCode(Yii::$app->session->get('dateFormat')), 'message' => 'Incorrect Interest Free Period End Date format.'],
            ['frequency', 'required'],
            ['is_debt_active', 'integer']
        ];

        // sanitize
        $filterRules = [
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'strip_tags'],
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'trim'],
            [['amount', 'payment'], 'filter', 'filter' => ['backend\components\helpers\Formatter', 'currencyToDecimal']],
            [['date', 'next_due_date', 'interest_free_end_date'], 'filter', 'filter' => ['backend\components\helpers\Formatter', 'appDateToPhpDate'], 'when' => function($model, $attribute) {
                return !empty($model->{$attribute});
            }]
        ];

        return array_merge($this->dateRules(), $this->nextDueDateRules(), $this->interestFreeEndDateRules(), $validationRules, $filterRules);

    }

    /**
     * Check whether debt name is allowed.
     * @param string $attribute
     */
    public function debtNameValid($attribute) {

        if ($this->frequency == 'one-time') {

            $debt = Debt::find()
                ->where([
                    'and',
                    ['budget_id' => $this->budget_id],
                    ['!=', 'id', $this->debt_id],
                    ['name' => $this->{$attribute}],
                    ['!=', 'frequency', 'one-time'],
                    ['archived' => 0]
                ])
                ->one();

        }
        else {

            $debt = Debt::find()
                ->where([
                    'and',
                    ['budget_id' => $this->budget_id],
                    ['!=', 'id', $this->debt_id],
                    ['name' => $this->{$attribute}],
                    ['archived' => 0]
                ])
                ->one();

        }

        if ($debt) {

            $this->addError($attribute, 'Please enter a unique debt name.');

        }

    }


    /**
     * Check if the account ID can be accessed by this user
     */
    public function accountIdValid($attribute) {

        $account = Account::findOne([
            'budget_id' => $this->budget_id,
            'id' => $this->{$attribute}
        ]);

        if (!$account) {

            $this->addError($attribute, 'Invalid account ID.');

        }

    }

}
