<?php

namespace backend\models\form;

use backend\components\validators\Currency;
use backend\models\form\common\DateField;
use Yii;

use backend\models\db\Account;
use backend\models\db\Income;

use backend\components\helpers\Formatter;
use backend\components\validators\DateStart;

/**
 * New jar form
 */
class AddIncome extends \yii\base\Model
{
    use DateField;

    public $name;
    public $description;
    public $amount;
    public $date;
    public $start_date;
    public $frequency;
    public $account_type_id;
    public $budget_id;
    public $account_id;
    public $income_id = -1;
    public $is_unbudgeted = 0;
    public $jar_id;
    public $adjust_start_date = false;

    public function rules()
    {
        // validate input
        $validationRules = [
            ['name', 'required', 'except' => 'update'],
            ['name', 'string', 'min' => 3],
            ['name', 'incomeNameValid'],
            ['amount', 'required'],
            ['amount', Currency::className(), 'min' => '0.01'],
            ['account_id', 'required', 'message' => 'Please select an account.'],
            ['account_id', 'integer'],
            ['account_id', 'accountIdValid'],
            ['date', 'required'],
            ['date', DateStart::className(), 'fieldName' => 'income'],
            ['date', 'required', 'message' => 'Please select a start date.', 'when' => function($model) {
                return ($model->income_id != -1);
            }],
            ['date', 'date', 'format' => Formatter::phpDateCodeToDateCode(Yii::$app->session->get('dateFormat')), 'message' => 'Incorrect date format.', 'except' => 'update'],
            ['frequency', 'required', 'except' => 'update'],
            ['jar_id', 'integer'],
            [['description', 'is_unbudgeted'], 'safe'],

            [['date', 'description', 'amount', 'account_id', 'jar_id'], 'safe', 'on' => 'update'],
            //['jar_id', 'required', 'on' => 'update']
        ];

        // sanitize
        $filterRules = [
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'strip_tags'],
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'trim'],
            ['amount', 'filter', 'filter' => ['backend\components\helpers\Formatter', 'currencyToDecimal']],
            [['start_date', 'date'], 'filter', 'filter' => ['backend\components\helpers\Formatter', 'appDateToPhpDate'], 'when' => function($model, $attribute) {
                return !empty($model->{$attribute});
            }]
        ];
        return array_merge($this->dateRules(), $validationRules, $filterRules);
    }

    /**
     * Check whether income name is allowed.
     * @param string $attribute
     */
    public function incomeNameValid($attribute)
    {
        if ($this->frequency == 'one-time') {
            $income = Income::find()
                ->where([
                    'and',
                    ['budget_id' => $this->budget_id],
                    ['!=', 'id', $this->income_id],
                    ['name' => $this->{$attribute}],
                    ['!=', 'frequency', 'one-time']
                ])
                ->one();
        }
        else {
            $income = Income::find()
                ->where([
                    'and',
                    ['budget_id' => $this->budget_id],
                    ['!=', 'id', $this->income_id],
                    ['name' => $this->{$attribute}],
                ])
                ->one();
        }

        if ($income) {
            $this->addError($attribute, 'Please enter a unique income name.');
        }
    }

    /**
     * Check if the account ID can be accessed by this user
     */
    public function accountIdValid($attribute)
    {
        $account = Account::findOne([
            'budget_id' => $this->budget_id,
            'id' => $this->{$attribute}
        ]);

        if (!$account) {
            $this->addError($attribute, 'Invalid account ID.');
        }
    }
}
