<?php
namespace backend\models\form;

use backend\components\helpers\Formatter;
use backend\components\helpers\Frequencies;
use backend\models\db\TimeSlot;
use backend\models\db\TimeSlotItem;
use backend\models\db\TimeZone;
use backend\models\form\common\EndDateField;
use backend\models\form\common\RepeatUntilDateField;
use backend\models\form\common\StartDateField;


/**
 * Add Time Slot form
 */
class AddTimeSlot extends \yii\base\Model
{
    use StartDateField, EndDateField, RepeatUntilDateField;

    const SINGLE_APPOINTMENT = 1;
    const SLOT_MINUTES = 2;

    public $name;
    public $startDate;
    public $startTime;
    public $endDate;
    public $endTime;
    public $repeat = false;
    public $frequency;
    public $repeatEvery = 1;
    public $slotType = self::SINGLE_APPOINTMENT;
    public $slotLength = 90;
    public $repeatUntilDate;

    public function rules() {
        // validate input
        $validationRules = [
            [['startDate', 'startTime', 'endDate', 'endTime'], 'required'],
            [['startDate', 'startTime', 'endDate', 'endTime'], 'datesCorrect'],
            ['repeatUntilDate', 'repeatUntilDateCorrect'],
            [['name', 'repeat', 'frequency', 'repeatEvery', 'slotType', 'slotLength', 'repeatUntilDate'], 'safe']
        ];
        // sanitize
        $filterRules = [
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'strip_tags'],
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'trim'],
            [['startDate', 'endDate', 'repeatUntilDate'], 'filter', 'filter' => ['backend\components\helpers\Formatter', 'appDateToPhpDate'], 'when' => function($model, $attribute) {
                return !empty($model->{$attribute});
            }]
        ];

        return array_merge($this->startDateRules(), $this->endDateRules(), $this->repeatUntilDateRules(), $validationRules, $filterRules);
    }


    public function datesCorrect($attribute)
    {
        $fromDate = Formatter::appDateToPhpDate($this->startDate) . ' ' . $this->startTime;
        $toDate = Formatter::appDateToPhpDate($this->endDate) . ' ' . $this->endTime;

        if ($fromDate >= $toDate) {
            $this->addError('startDate', 'End Date/Time must be after Start Date/Time.');
        }
        // TODO: check if timeslots are overlapping
        /*$timeSlot = TimeSlotItem::find()
            ->where(['and',
                []
            ])
            ->one();*/
    }

    public function repeatUntilDateCorrect($attribute)
    {
        if (Formatter::appDateToPhpDate($this->endDate) > Formatter::appDateToPhpDate($this->repeatUntilDate)) {
            $this->addError('repeatUntilDate', 'Repeat Until Date must be greater than or equal End Date/Time');
        }
    }

    public function loadFromDb(TimeSlot $slot, TimeZone $timeZone)
    {
        $startDateTime = $slot->start_date . ' ' . $slot->start_time;
        $endDateTime = $slot->end_date . ' ' . $slot->end_time;

        $this->name = $slot->name;
        $this->startDate = Formatter::utcDatetimeToLocalDatetime($startDateTime, $timeZone->code, 'Y-m-d');
        $this->startTime = Formatter::utcDatetimeToLocalDatetime($startDateTime, $timeZone->code, 'H:i');
        $this->endDate = Formatter::utcDatetimeToLocalDatetime($endDateTime, $timeZone->code, 'Y-m-d');
        $this->endTime = Formatter::utcDatetimeToLocalDatetime($endDateTime, $timeZone->code, 'H:i');
        $this->repeat = $slot->repeat;
        $this->frequency = $slot->frequency;
        $this->repeatEvery = $slot->repeat_every;
        $this->repeatUntilDate = $slot->repeat_until_date ?
            Formatter::utcDatetimeToLocalDatetime($slot->repeat_until_date, $timeZone->code, 'Y-m-d') :
            Formatter::utcDatetimeToLocalDatetime(strtotime('+1 month', strtotime($startDateTime)), $timeZone->code, 'Y-m-d');
    }

    public function getTextSummary(TimeZone $timeZone)
    {
        $startDate = '<strong>' . Formatter::date($this->startDate) . '</strong>';
        $endDate = '<strong>' . Formatter::date($this->endDate) . '</strong>';
        $untilDate = '<strong>' . Formatter::date($this->repeatUntilDate) . '</strong>';
        $startTime = '<strong>' . $this->startTime . '</strong>';
        $endTime = '<strong>' . $this->endTime . '</strong>';

        if ($this->repeat) {
            switch ($this->frequency) {
                case 'weekly':
                    $dateDetail = 'on <strong>' . date('l', strtotime($this->startDate)) . '</strong>';
                    break;
                case 'monthly':
                    $dateDetail = 'on <strong>day ' . date('j', strtotime($this->startDate)) . '</strong>';
                    break;
                case 'annually':
                    $dateDetail = 'on <strong>' . date('F j', strtotime($this->startDate)) . '</strong>';
                    break;
                default:
                    $dateDetail = '';
            }
            $lengthTime = (strtotime($this->endDate . ' ' . $this->endTime) -
                strtotime($this->startDate . ' ' . $this->startTime)) / 60;
            $lasting = $lengthTime > 120 ?
                floor($lengthTime / 60) . ' hours ' . ($lengthTime % 60 != 0 ? ($lengthTime % 60) . ' minutes' : '') :
                $lengthTime . ' minutes';
            if ($this->repeatEvery > 1) {
                $names = Frequencies::getUniqueNames();
                $name = '<strong>' . $this->repeatEvery . ' ' . strtolower($names[$this->frequency]) . '</strong>';
                $summary =
                    'Every ' . $name . ' ' . $dateDetail . ' at ' . $startTime . '<br/>' .
                    'Lasting <strong>' . $lasting . '</strong><br/>' .
                    'Starting from ' . $startDate . ' until ' . $untilDate;
            }
            else {
                $frequencies = Frequencies::getUnique();
                $frequency = '<strong>' . $frequencies[$this->frequency] . '</strong>';
                $summary =
                    $frequency . ' ' . $dateDetail . ' at ' . $startTime . '<br/>' .
                    'Lasting <strong>' . $lasting . '</strong><br/>' .
                    'Starting from ' . $startDate . ' until ' . $untilDate;
            }
        }
        else {
            if ($this->startDate == $this->endDate) {
                $summary =
                    'On ' . $startDate . ' (' . date('l', strtotime($this->startDate)) . ')<br/>' .
                    'From ' . $startTime . ' to ' . $endTime;
            }
            else {
                $summary =
                    'From ' . $startDate . ' (' . date('l', strtotime($this->startDate)) . ') ' .
                    $startTime . '<br/>' .
                    'To ' . $endDate . ' (' . date('l', strtotime($this->endDate)) . ') ' .
                    $endTime;
            }
        }
        return $summary . '<br/>' .
            $timeZone->name . ' time';
    }

}
