<?php

namespace backend\models\form;

use backend\components\validators\Currency;
use backend\models\db\Jar;
use backend\models\form\common\DateField;
use Yii;

use backend\models\db\Account;
use backend\models\db\Expense;

use backend\components\helpers\Formatter;
use backend\components\validators\DateStart;

/**
 * New transaction form
 */
class AddTransaction extends \yii\base\Model
{
    use DateField;

    public $jar_id;
    public $account_id;
    public $debt_id = false;
    public $expense_id;
    public $amount;
    public $date;
    public $budget_id = -1;
    public $description;
    public $tax_deductible = 0;

    public function rules()
    {
        // validate input
        $validationRules = [
            ['account_id', 'required', 'message' => 'Please select an account.'],
            ['account_id', 'accountIdValid'],
            ['amount', 'required'],
            ['amount', Currency::className(), 'min' => '0.01'],
            ['date', 'required'],
            ['date', DateStart::className(), 'fieldName' => 'transaction'],
            ['date', 'date', 'format' => Formatter::phpDateCodeToDateCode(Yii::$app->session->get('dateFormat')), 'message' => 'Incorrect date format.'],
            [['expense_id', 'tax_deductible'], 'integer'],
            ['expense_id', 'expenseIdValid'],
            ['description', 'string', 'max' => 255],
            ['jar_id', 'required'],
            ['debt_id', 'safe'],
            ['debt_id', 'required', 'when' => function($model, $attribute) {
                $jar = Jar::findOne($model->jar_id);
                return $jar->jar_type_id != 1;
            }, 'whenClient' => "function (attribute, value) {
                return $(\"#add-transaction-form #addtransaction-jar_id option[value='\" + $(\"#add-transaction-form #addtransaction-jar_id\").val() + \"']\").data('debt') == 1;
            }"],

            [['date', 'description', 'amount', 'account_id', 'jar_id', 'tax_deductible'], 'safe', 'on' => 'update']
        ];

        // sanitize
        $filterRules = [
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'strip_tags'],
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'trim'],
            ['amount', 'filter', 'filter' => ['backend\components\helpers\Formatter', 'currencyToDecimal']],
            ['date', 'filter', 'filter' => ['backend\components\helpers\Formatter', 'appDateToPhpDate'], 'when' => function($model, $attribute) {
                return !empty($model->{$attribute});
            }]
        ];

        return array_merge($this->dateRules(), $validationRules, $filterRules);
    }

    /**
     * Check if the account ID can be accessed by this user
     */
    public function accountIdValid($attribute)
    {
        $account = Account::findOne([
            'budget_id' => $this->budget_id,
            'id' => $this->{$attribute}
        ]);
        if (!$account) {
            $this->addError($attribute, 'Invalid account selected.');
        }
    }

    /**
     * Check if the account ID can be accessed by this user
     */
    public function expenseIdValid($attribute)
    {
        if (!empty($this->{$attribute})) {
            $expense = Expense::findOne(['id' => $this->{$attribute}]);
            if (!$expense || !Yii::$app->user->can('editJar', ['jarId' => $expense->jar_id])) {
                $this->addError($attribute, 'This expense cannot be used.');
            }
        }
    }

}
