<?php
namespace backend\modules\admin\controllers;

use backend\components\helpers\Formatter;
use backend\components\helpers\Frequencies;
use backend\components\helpers\JsonTools;
use backend\models\db\TimeSlot;
use backend\models\db\TimeSlotBooking;
use backend\models\db\TimeSlotItem;
use backend\models\db\TimeZone;
use backend\models\db\User;
use backend\models\db\UserMeta;
use backend\models\form\AddAppointment;
use backend\models\form\AddTimeSlot;
use dosamigos\typeahead\Bloodhound;
use Yii;

use backend\models\db\Appointment;
use backend\models\db\AppointmentUser;

use yii\filters\AccessControl;
use backend\components\CustomController;
use yii\helpers\ArrayHelper;
use yii\helpers\Json;
use yii\helpers\Url;
use yii\web\Response;


class CoachController extends CustomController
{
    public $enableCsrfValidation = false;

    public function behaviors() {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'conversation-detail',
                            'appointments',
                            'events',
                            'bookings',
                            'add-time-slot',
                            'delete-time-slot',
                            'summary',
                            'add-appointment',
                            'delete-appointment',
                            'hours-list'
                        ],
                        'allow' => true,
                        'roles' => ['coach'],
                    ]
                ],
            ]
        ];
    }


    public function actionIndex()
    {
        $user = $this->user;
        $clients = $user->getConversationClients();

        $this->view->title = 'Communication';
        return $this->render('index', [
            'user' => $user,
            'timezone' => $user->getTimeZone(),
            'clients' => $clients
        ]);
    }

    public function actionConversationDetail()
    {
        $userId = (int)Yii::$app->getRequest()->getQueryParam('client_id', 0);
        $conversationUser = User::findOne($userId);
        $conversation = $conversationUser->getMainConversation();
        $user = $this->user;
        $clients = $user->getConversationClients();

        $messages = $conversation->getMessages($user->id, 25, false, true);
        $conversation->markAsRead($user->id, $messages);

        $this->view->title = 'Conversation with ' . $conversationUser->getFullName();
        return $this->render('conversation-detail', [
            'user' => $user,
            'timezone' => $user->getTimeZone(),
            'clients' => $clients,
            'messages' => $messages,
            'messagesData' => $conversation->getData(),
            'conversation' => $conversation,
            'userId' => $userId
        ]);
    }

    public function actionAppointments()
    {
        $this->view->title = 'Appointments';
        return $this->render('appointments');
    }

    public function actionEvents()
    {
        $start = date('Y-m-d', strtotime(Yii::$app->request->getQueryParam('start', date('Y-m-d'))));
        $end   = date('Y-m-d', strtotime(Yii::$app->request->getQueryParam('end', date('Y-m-d'))));
        $startTime = $start . ' 00:00:00';
        $endTime   = $end . ' 23:59:59';
        $timezone = $this->user->getTimeZone();

        $events = TimeSlotItem::find()
            ->joinWith('timeSlot')
            ->where(['and',
                ['coach_id' => Yii::$app->user->identity->id],
                ['between', 'start_at', $startTime, $endTime],
                ['between', 'end_at', $startTime, $endTime]
            ])
            ->all();

        $result = [];
        /** @var TimeSlotItem $event */
        foreach ($events as $event) {
            $result[] = [
                'id' => $event->id,
                'type' => TimeSlotItem::className(),
                'title' => $event->timeSlot->name,
                'allDay' => false, // TODO: check for an all-day event
                'start' => Formatter::utcDatetimeToLocalDatetime($event->start_at, $timezone->code, 'c'),
                'end' => Formatter::utcDatetimeToLocalDatetime($event->end_at, $timezone->code, 'c'),
                'color' => '#7f66b3',
                'overlap' => false,
                'editable' => false,
                'durationEditable' => false
            ];
        }

        // users' bookings
        $result = array_merge($result, TimeSlotBooking::getUserBookings(
            null,
            $startTime,
            $endTime,
            $timezone,
            false,
            $this->user->id
        ));

        Yii::$app->response->format = Response::FORMAT_JSON;
        return $result;
    }

    public function actionBookings()
    {
        $start = date('Y-m-d', strtotime(Yii::$app->request->getQueryParam('start', date('Y-m-d'))));
        $end   = date('Y-m-d', strtotime(Yii::$app->request->getQueryParam('end', date('Y-m-d'))));
        $startTime = $start . ' 00:00:00';
        $endTime   = $end . ' 23:59:59';
        $timezone  = $this->user->getTimeZone();

        $userEvents = TimeSlotBooking::find()
            ->joinWith('timeSlotItem.timeSlot')
            ->where(['and',
                ['coach_id' => $this->user->id],
                ['between', TimeSlotBooking::tableName() . '.start_at', $startTime, $endTime],
                ['between', TimeSlotBooking::tableName() . '.end_at', $startTime, $endTime]
            ])
            ->all();
        $result = [];
        /** @var TimeSlotBooking $event */
        foreach ($userEvents as $event) {
            $result[] = [
                'id' => $event->id,
                'type' => TimeSlotBooking::className(),
                'title' => $event->user->fullName,
                'allDay' => false,
                'start' => Formatter::utcDatetimeToLocalDatetime($event->start_at, $timezone->code, 'c'),
                'end' => Formatter::utcDatetimeToLocalDatetime($event->end_at, $timezone->code, 'c'),
                'url' => Url::to(['/admin/user/switch', 'id' => $event->user_id]),
                'editable' => true
            ];
        }
        Yii::$app->response->format = Response::FORMAT_JSON;
        return $result;
    }

    public function actionDeleteTimeSlot()
    {
        $id = Yii::$app->request->getQueryParam('id');
        $all = Yii::$app->request->getQueryParam('all');
        $confirm = Yii::$app->request->getQueryParam('confirm');
        $timezone = $this->user->getTimeZone();

        $item = TimeSlotItem::findOne($id);
        if (!$item || !$id) {
            echo JsonTools::errorMessage('Incorrect time slot ID provided.');
            die();
        }

        if ($confirm) {
            $timeSlot = $item->timeSlot;
            if ($all) {
                // send notifications to any users who have already booked an appointment for the deleted time slot
                $bookings = TimeSlotBooking::find()
                    ->where(['and',
                        ['time_slot_id' => $item->time_slot_id],
                        ['>=', 'start_at', $item->start_at]
                    ])
                    ->all();
                /** @var TimeSlotBooking $booking */
                foreach ($bookings as $booking) {
                    $booking->sendCancelledNotification($this->user, $timezone);
                }

                TimeSlotBooking::deleteAll(['and',
                    ['time_slot_id' => $item->time_slot_id],
                    ['>=', 'start_at', $item->start_at]
                ]);
                $timeSlot->repeat_until_date = date('Y-m-d', strtotime('-1 day', strtotime($item->start_at)));
                $timeSlot->save();
                TimeSlotItem::deleteAll(['and',
                    ['time_slot_id' => $item->time_slot_id],
                    ['>=', 'start_at', $item->start_at]
                ]);
                if (count($timeSlot->timeSlotItems) <= 0) {
                    $timeSlot->delete();
                }
                echo JsonTools::successMessage('Selected time slot and all future time slots successfully deleted.');
            }
            else {
                TimeSlotBooking::deleteAll(['time_slot_item_id' => $item->id]);
                $item->delete();
                $timeSlot->delete();
                echo JsonTools::successMessage('Selected time slot successfully deleted.');
            }
            die();
        }


        $this->view->title = 'Confirm Deletion';
        return $this->render('confirm-delete', [
            'item' => $item,
            'itemDate' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($item->start_at, $timezone->code,'Y-m-d H:i:s')),
            'timezone' => $timezone
        ]);
    }

    public function actionAddAppointment()
    {
        $appointmentForm = new AddAppointment();
        $timezone = $this->user->getTimeZone();

        if (Yii::$app->request->isPost && $appointmentForm->load(Yii::$app->request->post())) {
            if (!$appointmentForm->validate()) {
                echo JsonTools::formErrorMessage($appointmentForm);
                die();
            }
            $slotBooking = new TimeSlotBooking();
            $slotBooking->setAttributes([
                'time_slot_id' => $appointmentForm->getTimeSlot()->time_slot_id,
                'time_slot_item_id' => $appointmentForm->timeSlotItemId,
                'user_id' => $appointmentForm->userId,
                'start_at' => Formatter::localDatetimeToUtcDatetime($appointmentForm->startDateTime, 'Y-m-d H:i:s'),
                'end_at' => Formatter::localDatetimeToUtcDatetime($appointmentForm->getEndDateTime(), 'Y-m-d H:i:s')
            ]);
            if ($slotBooking->save()) {
                $slotBooking->sendCreatedNotification($this->user, $timezone);
                echo JsonTools::successMessage('Appointment successfully added.');
            } else {
                echo JsonTools::errorMessage('Could not add the appointment.');
            }
            die();
        }

        $hours = [];
        $slots = TimeSlotItem::getTimeSlotItems($this->user->id, $timezone);
        $slotObjects = TimeSlotItem::getTimeSlotItems($this->user->id, $timezone, null, false);
        $firstItem = !empty($slotObjects) ? current($slotObjects) : null;
        if ($firstItem) {
            $hours = $firstItem->getPossibleHoursList($timezone);
        }

        $this->view->title = 'Add Appointment';
        return $this->render('add-appointment', [
            'formModel' => $appointmentForm,
            'availableSlots' => $slots,
            'hours' => $hours,
            'users' => ArrayHelper::map($this->user->getCoachUsers(), 'id', 'fullName')
        ]);
    }

    public function actionHoursList()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        $id = Yii::$app->request->getQueryParam('id');
        $timeSlotItem = TimeSlotItem::findOne($id);
        if ($timeSlotItem) {
            return $timeSlotItem->getPossibleHoursList($this->user->getTimeZone());
        }
        return [];
    }

    public function actionAddTimeSlot()
    {
        $id = Yii::$app->request->getQueryParam('id');
        $type = Yii::$app->request->getQueryParam('type');
        $timezone = $this->user->getTimeZone();

        if ($type == TimeSlotBooking::className()) {
            return $this->handleTimeSlotBookingClick($id, $timezone);
        }
        else {
            return $this->handleTimeSlotItemClick($id, $timezone);
        }
    }

    public function actionDeleteAppointment()
    {
        return $this->handleTimeSlotBookingClick(
            Yii::$app->request->getQueryParam('id'),
            $this->user->getTimeZone()
        );
    }

    protected function handleTimeSlotBookingClick($id, $timezone)
    {
        $confirm = Yii::$app->request->getQueryParam('confirm');

        $item = TimeSlotBooking::findOne($id);
        if (!$item || !$id) {
            echo JsonTools::errorMessage('Incorrect booking ID provided.');
            die();
        }

        if ($confirm) {
            // send notifications to any users who have already booked an appointment for the deleted time slot
            $item->sendCancelledNotification($this->user, $timezone);
            $item->delete();
            echo JsonTools::successMessage('Selected booking successfully deleted.');
            die();
        }

        $this->view->title = 'Confirm Deletion';
        return $this->render('confirm-appointment-delete', [
            'item' => $item,
            'itemDate' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($item->start_at, $timezone->code,'Y-m-d H:i:s')),
            'timezone' => $timezone
        ]);
    }

    protected function handleTimeSlotItemClick($id, $timezone)
    {
        $addTimeSlot = new AddTimeSlot();
        if ($id) {
            $item = TimeSlotItem::findOne($id);
            if (!$item) {
                echo JsonTools::errorMessage('Incorrect time slot ID provided.');
                die();
            }
        }

        if (Yii::$app->request->isPost && $addTimeSlot->load(Yii::$app->request->post())) {
            if (!$addTimeSlot->validate()) {
                echo JsonTools::formErrorMessage($addTimeSlot);
                die();
            }
            if ($id) {
                if ($this->editTimeSlot($addTimeSlot, $item)) {
                    echo JsonTools::successMessage('Time Slot successfully updated.');
                } else {
                    echo JsonTools::errorMessage('Could not update Time Slot.');
                }
                die();
            }
            else {
                if ($this->addTimeSlot($addTimeSlot)) {
                    echo JsonTools::successMessage('Time Slot successfully saved.');
                } else {
                    echo JsonTools::errorMessage('Could not save Time Slot.');
                }
                die();
            }
        }

        if ($id) {
            $addTimeSlot->loadFromDb($item->timeSlot, $timezone);
            // update start and end dates to reflect the selected event
            $addTimeSlot->startDate = Formatter::utcDatetimeToLocalDatetime($item->start_at, $timezone->code, 'Y-m-d');
            $addTimeSlot->endDate = Formatter::utcDatetimeToLocalDatetime($item->end_at, $timezone->code, 'Y-m-d');
        }
        else {
            $start = (int)Yii::$app->request->getQueryParam('start') / 1000;
            $end = (int)Yii::$app->request->getQueryParam('end') / 1000;
            if ($start) {
                $addTimeSlot->startDate = Formatter::localDatetimeToUtcDatetime($start, 'Y-m-d');
                $addTimeSlot->startTime = Formatter::localDatetimeToUtcDatetime($start, 'H:i');
                $addTimeSlot->repeatUntilDate = Formatter::localDatetimeToUtcDatetime(strtotime('+1 month', $start), 'Y-m-d');
            }
            if ($end) {
                $addTimeSlot->endDate = Formatter::localDatetimeToUtcDatetime($end, 'Y-m-d');
                $addTimeSlot->endTime = Formatter::localDatetimeToUtcDatetime($end, 'H:i');
            }
        }

        /*$engine = new Bloodhound([
            'name' => 'timeEngine',
            'clientOptions' => [
                'datumTokenizer' => new \yii\web\JsExpression("Bloodhound.tokenizers.whitespace"),
                'queryTokenizer' => new \yii\web\JsExpression("Bloodhound.tokenizers.whitespace"),
                'local' => [
                    '1:00am',
                    '1:30am'
                ]
            ]
        ]);*/

        $this->view->title = ($id ? 'Edit' : 'Add') . ' Time Slot';
        return $this->render('add-time-slot', [
            'edit' => (int)$id,
            'formModel' => $addTimeSlot,
            'frequencies' => Frequencies::getUnique(),
            'timezone' => $timezone
            //'engine' => $engine
        ]);
    }

    public function actionSummary()
    {
        $text = '';
        $timeSlot = new AddTimeSlot();
        if ($timeSlot->load(Yii::$app->request->post())) {
            $meta = UserMeta::findAllSorted(['user_id' => $this->user->id]);
            $timezone = TimeZone::findOne((isset($meta['time_zone_id'])) ? $meta['time_zone_id'] : Yii::$app->params['defaultTimeZoneId']);

            $text = $timeSlot->getTextSummary($timezone);
        }
        Yii::$app->response->format = Response::FORMAT_JSON;
        return [
            'text' => $text
        ];
    }

    protected function addTimeSlot(AddTimeSlot $addTimeSlot)
    {
        $localStartDateTime = $addTimeSlot->startDate . ' ' . $addTimeSlot->startTime;
        $localEndDateTime = $addTimeSlot->endDate . ' ' . $addTimeSlot->endTime;
        $utcStart = Formatter::localDatetimeToUtcDatetime($localStartDateTime);
        $utcEnd = Formatter::localDatetimeToUtcDatetime($localEndDateTime);
        $timeSlot = new TimeSlot();
        $timeSlot->setAttributes([
            'coach_id' => $this->user->id,
            'start_date' => substr($utcStart, 0, 10),
            'end_date' => substr($utcEnd, 0, 10),
            'start_time' => substr($utcStart, 11, 5),
            'end_time' => substr($utcEnd, 11, 5),
            'repeat' => $addTimeSlot->repeat,
            'name' => $addTimeSlot->name,
            'frequency' => $addTimeSlot->repeat ? $addTimeSlot->frequency : null,
            'repeat_every' => $addTimeSlot->repeatEvery,
            'repeat_until_date' => $addTimeSlot->repeat ? $addTimeSlot->repeatUntilDate : null
        ]);
        return $timeSlot->save() && $timeSlot->generateSlotItems($localStartDateTime, $localEndDateTime);
    }

    protected function editTimeSlot(AddTimeSlot $addTimeSlot, TimeSlotItem $item)
    {
        // TODO: show warning for the coach if there are already users who have booked in the time slot that is about to be edited
        // TODO: send notifications to any users who have already booked an appointment for the edited time slot

        $timeSlot = $item->timeSlot;
        // editing time slot is done in 2 phases
        if ($timeSlot->repeat) {
            // the existing time slot will no longer cover future dates
            $timeSlot->repeat_until_date = date('Y-m-d', strtotime('-1 day', strtotime($item->start_at)));
            $timeSlot->save();
            // 1st phase - delete existing time slot entries
            TimeSlotItem::deleteAll(['and',
                ['time_slot_id' => $item->time_slot_id],
                ['>=', 'start_at', $item->start_at]
            ]);
            // delete parent event if there are no child items left
            if (count($timeSlot->timeSlotItems) <= 0) {
                $timeSlot->delete();
            }
        }
        else {
            // for one-off events the parent time-slot doesn't make sense without the child event
            $item->delete();
            $timeSlot->delete();
        }

        // 2nd phase - create (clone) the time slot to a new one and update the old one
        return $this->addTimeSlot($addTimeSlot);
    }

}
