<?php

namespace backend\modules\admin\controllers;

use Yii;

use backend\components\CustomController;

use yii\helpers\Url;
use yii\filters\AccessControl;
use yii\web\UploadedFile;

use backend\models\db\Note;
use backend\models\db\NoteAttachment;
use backend\models\db\User;

use backend\modules\admin\models\form\AddNote;

use yii\helpers\FileHelper;
use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\JsonTools;


class NotesController extends CustomController
{

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'client-detail',
                            'add-note',
                            'edit-note',
                            'notes-table',
                            'remove-note-attachment',
                            'remove-note'
                        ],
                        'allow' => true,
                        'roles' => ['coach']
                    ]
                ],
            ]
        ];

    }


    public function actionIndex()
    {
        $clients = [];
        if ($this->user->user_role_id == 2) {
            $user = User::findOne($this->user->id);
            $clients = $user->getClients(true);
        }

        $this->view->title = 'Notes';
        return $this->render('index', [
            'clients' => $clients,
        ]);
    }


    public function actionClientDetail() {

        $clientId = Yii::$app->getRequest()->getQueryParam('client_id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        $user = User::findOne($this->user->id);
        if (!is_numeric($clientId) || !$client = $user->getUserChildren()->where(['user_id' => $clientId])->one()->user) {

            return $this->accessError();

        }

        // get the notes
        $notes = $client->getNotes()
            ->where(['archived' => 0])
            ->all();

        // get all jars for sidebar
        $this->view->title = $client->getFullName(true) . ' notes';

        // switch view based on whether this is an initial page load
        if ($this->pageLoadRequest || $fullLoad) {

            $clients = $user->getClients(true);

            return $this->render('client-detail', [
                'clients' => $clients,
                'clientId' => $clientId,
                'notes' => $notes
            ]);

        }
        else {

            return $this->render('client-detail-content', [
                'clientId' => $clientId,
                'notes' => $notes
            ]);

        }

    }


    public function actionAddNote() {

        $clientId = Yii::$app->getRequest()->getQueryParam('client_id', false);
        $user = User::findOne($this->user->id);
        if (!is_numeric($clientId) || !$client = $user->getUserChildren()->where(['user_id' => $clientId])->one()->user) {

            return $this->accessError();

        }

        $noteForm = new AddNote();
        if ($noteForm->load(Yii::$app->request->post())) {

            $noteForm->attachment = UploadedFile::getInstances($noteForm, 'attachment');

            if ($noteForm->validate()) {

                $note = new Note();
                $note->setAttributes([
                    'name' => $noteForm->name,
                    'content' => $noteForm->content,
                    'user_id' => $user->id,
                    'user_client_id' => $client->id,
                    'time_created' => Formatter::localDatetimeToUtcDatetime(time()),
                    'time_edited' => Formatter::localDatetimeToUtcDatetime(time())
                ]);
                $note->save();

                // save attachments
                if (count($noteForm->attachment)) {

                    $path = 'uploads/note_attachments/' . $note->id . '/';

                    if (!file_exists($path)) {

                        FileHelper::createDirectory($path);

                    }

                    foreach ($noteForm->attachment as $a) {

                        $saved = $a->saveAs($path . $a->name);

                        if ($saved) {

                            $noteAttachment = new NoteAttachment();
                            $noteAttachment->setAttributes([
                                'path' => $path,
                                'name' => $a->name,
                                'note_id' => $note->id
                            ]);
                            $noteAttachment->save();

                        }
                        elseif (Yii::$app->request->isAjax) {

                            echo JsonTools::errorMessage('An error occured while saving attachments.');
                            die();

                        }

                    }

                }

                if (Yii::$app->request->isAjax) {

                    echo JsonTools::successMessage('Note successfully created.');
                    die();

                }

            }
            elseif (Yii::$app->request->isAjax) {

                echo JsonTools::formErrorMessage($noteForm);
                die();

            }

        }
        else {

            $this->view->title = 'Add a Note';

            return $this->render('add-note', [
                'formModel' => $noteForm,
                'clientId' => $clientId
            ]);

        }

    }


    public function actionEditNote() {

        $noteId = Yii::$app->getRequest()->getQueryParam('id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        $user = User::findOne($this->user->id);
        $note = Note::findOne($noteId);

        if (!$note || ($note->user_id != $user->id)) {

            return $this->accessError();

        }

        $noteForm = new AddNote();
        if ($noteForm->load(Yii::$app->request->post())) {

            $noteForm->attachment = UploadedFile::getInstances($noteForm, 'attachment');

            if ($noteForm->validate()) {

                // save note
                $note->setAttributes($noteForm->getAttributes());
                $note->time_edited = Formatter::localDatetimeToUtcDatetime(time());
                $note->save();

                // save attachments
                if (count($noteForm->attachment)) {

                    $path = 'uploads/note_attachments/' . $note->id . '/';

                    if (!file_exists($path)) {

                        FileHelper::createDirectory($path);

                    }

                    foreach ($noteForm->attachment as $a) {

                        $saved = $a->saveAs($path . $a->name);

                        if ($saved) {

                            $noteAttachment = new NoteAttachment();
                            $noteAttachment->setAttributes([
                                'path' => $path,
                                'name' => $a->name,
                                'note_id' => $note->id
                            ]);
                            $noteAttachment->save();

                        }
                        elseif (Yii::$app->request->isAjax) {

                            echo JsonTools::errorMessage('An error occured while saving attachments.');
                            die();

                        }

                    }

                }

                if (Yii::$app->request->isAjax) {

                    echo JsonTools::successMessage('Note successfully saved.');
                    die();

                }

            }
            elseif (Yii::$app->request->isAjax) {

                echo JsonTools::formErrorMessage($noteForm);
                die();

            }

        }
        else {

            $noteForm->setAttributes($note->getAttributes());

            $attachments = NoteAttachment::find()
                ->where(['note_id' => $note->id])
                ->orderBy('name asc')
                ->all();

            $this->view->title = 'Note Detail';

            return $this->render('edit-note', [
                'formModel' => $noteForm,
                'noteId' => $note->id,
                'attachments' => $attachments
            ]);

        }
    }


    public function actionNotesTable() {

        $clientId = Yii::$app->getRequest()->getQueryParam('client_id', false);
        //$fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        $user   = User::findOne($this->user->id);
        $child  = $user->getUserChildren()->where(['user_id' => $clientId])->one();
        if (!is_numeric($clientId) || !$child || !$child->user) {
            return $this->accessError();
        }
        $client = $child->user;

        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => Note::tableName() . '.name'
            ],
            [
                'type' => 'normal',
                'as' => 'time_created',
                'select' => Note::tableName() . '.time_created'
            ],
            [
                'type' => 'normal',
                'as' => 'time_edited',
                'select' => Note::tableName() . '.time_edited'
            ]
        ];


        $notesQuery = Note::find()
            ->select([Note::tableName() . '.name', 'time_created', 'time_edited', 'IFNULL(note_id,' . Note::tableName() . '.id) AS na_note_id',
                'count(note_id) as attachments_count', Note::tableName() . ".id"])
            ->leftJoin(NoteAttachment::tableName(), NoteAttachment::tableName() . ".note_id = " . Note::tableName() . ".id")
            ->where([
                'AND',
                ['user_client_id' => $client->id],
                ['archived' => 0]
            ])
            // grouping by all selected fields to meet the default behavior of MySQL 5.7.5 (which is the preferred behavior)
            ->groupBy(['na_note_id', 'name', 'time_created', 'time_edited', Note::tableName() . '.id']);

        $responseData = DataTables::getResponseData($tableColumns, $notesQuery, $notesQuery);

        foreach ($responseData['items'] as $i) {

            $responseData['response']['data'][] = [
                $i->name,
                Formatter::datetime($i->time_created),
                Formatter::datetime($i->time_edited),
                $i->attachments_count,
                '<div class="text-center">
                    <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                    <ul class="dropdown-menu" aria-labelledby="menu1">
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['/admin/notes/edit-note', 'id' => $i->id]) . '"><i class="glyphicon glyphicon-eye-open"></i>View</a>
                        </li>
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['/admin/notes/remove-note', 'id' => $i->id]) . '"><i class="glyphicon glyphicon-trash"></i>Delete</a>
                        </li>
                    </ul>
                </div>'
            ];

        }

        echo json_encode($responseData['response']);
        die();

    }


    public function actionRemoveNoteAttachment() {

        $attachmentId = Yii::$app->getRequest()->getQueryParam('note_attachment_id', false);
        $attachment = NoteAttachment::findOne($attachmentId);

        if ($attachment) {

            $note = $attachment->getNote()
                ->where(['user_id' => Yii::$app->user->id])
                ->one();

            if ($note) {

                unlink($attachment->path . $attachment->name);
                $attachment->delete();

            }

        }

    }


    public function actionRemoveNote() {

        $noteId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        $user = User::findOne(Yii::$app->user->id);
        $note = Note::findOne($noteId);

        if (!$note || ($note->user_id != $user->id)) {

            return $this->accessError();

        }

        if ($confirm) {

            $note->archived = 1;
            $note->save();

            echo JsonTools::successMessage('Note successfully removed.');
            die();

        }

        $this->view->title = 'Remove Note';

        return $this->render('remove-note', [
            'noteId' => $note->id
        ]);

    }

}
