<?php

namespace common\models\form;

use backend\components\helpers\DripHelper;
use backend\components\helpers\Formatter;
use backend\components\helpers\Password;
use backend\models\db\User;
use backend\models\db\UserMeta;
use backend\models\db\UserParent;
use backend\models\db\UserPlan;
use backend\models\db\UserPlanType;
use backend\models\db\UserPlanVariation;
use backend\models\db\UserRegistration;
use backend\models\db\UserRole;
use Yii;
use yii\base\Model;
use yii\helpers\Url;

/**
 * Login form
 */
class Register extends Model {

    public $token;
    public $plan_variation_id;
    public $stripe_charge_id;
    public $username;
    public $email;
    public $password;
    public $password_repeat;
    public $first_name;
    public $last_name;
    public $address;
    public $coupon;
    public $source;
    public $webhookData;

    public $resetPasswordAfterRegistration = false;


    public function rules() {

        // validate input
        $validationRules = [
            ['username', 'string', 'min' => 8],
            ['username', 'match', 'not' => true, 'pattern' => '/ /', 'message' => 'Username may not contain spaces.'],
            ['username', 'validateBootcampUser'],
            ['email', 'email'],
            ['email', 'validateBootcampUser'],
            ['password', 'string', 'min' => 8],
            //['password', 'passwordValid'],
            ['password_repeat', 'compare', 'compareAttribute' => 'password'],
            ['password_repeat', 'safe'],

            [['username', 'email', 'password', 'password_repeat'], 'required', 'on' => [
                UserPlanType::PLAN_TYPE_FREE,
                UserPlanType::PLAN_TYPE_BOOTCAMP
            ]],
            [['username', 'email', 'password', 'password_repeat', 'first_name', 'last_name', 'address'], 'required', 'on' => [
                UserPlanType::PLAN_TYPE_FREE_12_MONTHS,
                UserPlanType::PLAN_TYPE_APP,
                UserPlanType::PLAN_TYPE_COACHING,
                UserPlanType::PLAN_TYPE_FINANCIAL_COACHING_PROGRAM,
                UserPlanType::PLAN_TYPE_MASTERING_MONEY_COACHING_PROGRAM,
                UserPlanType::PLAN_TYPE_KICKSTART,
                UserPlanType::PLAN_TYPE_FINANCIAL_KICKSTARTER,
                UserPlanType::PLAN_TYPE_MEMBERSHIP,
                UserPlanType::PLAN_TYPE_MAINTENANCE,
                UserPlanType::PLAN_TYPE_MAINTENANCE_OLD,
                UserPlanType::PLAN_TYPE_COACHING_MAINTENANCE,
                UserPlanType::PLAN_TYPE_MOSAIC,
                UserPlanType::PLAN_TYPE_STRONG_MONEY_SKILLS,
                UserPlanType::PLAN_TYPE_BOOTCAMP_ACCESS
            ]],
            [['username', 'email', 'password', 'password_repeat', 'first_name'], 'required', 'on' => [
                UserPlanType::PLAN_TYPE_CONQUER_YOUR_CC_3_MONTHS,
                UserPlanType::PLAN_TYPE_CONQUER_YOUR_CC_1_YEAR,
                UserPlanType::PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_3_MONTHS,
                UserPlanType::PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_1_YEAR
            ]],
            ['last_name', 'safe', 'on' => [
                UserPlanType::PLAN_TYPE_CONQUER_YOUR_CC_3_MONTHS,
                UserPlanType::PLAN_TYPE_CONQUER_YOUR_CC_1_YEAR,
                UserPlanType::PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_3_MONTHS,
                UserPlanType::PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_1_YEAR
            ]],
            [['token', 'plan_variation_id', 'coupon'], 'safe']
        ];

        // sanitize
        $filterRules = [
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'strip_tags'],
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'trim']
        ];

        return array_merge($validationRules, $filterRules);
    }

    public function validateBootcampUser($attribute)
    {
        // throw an error only when the user exists and is NOT on Bootcamp plan
        $bootcampUser = $this->checkBootcampUser($attribute);
        if ($bootcampUser === false) {
            $this->addError($attribute, "This {$attribute} has already been taken.");
        }
    }

    /**
     * Check whether password is correctly formated.
     * @param string $attribute
     */
    public function passwordValid($attribute) {

        $valid = true;

        // check if there's an uppercase character
        if (strtolower($this->$attribute) === $this->$attribute) {
            $valid = false;
        }

        // check if there's a numeric value in the string
        if (preg_match("/[0-9]+/", $this->$attribute) < 1) {
            $valid = false;
        }

        if (!$valid) {
            $this->addError($attribute, 'Password must have 1 number and 1 uppercase letter.');
        }
    }

    public function generateUsername()
    {
        $parts = [];
        if (!empty($this->first_name)) {
            $parts[] = preg_replace('/\s+/', '', strtolower($this->first_name));
        }
        if (!empty($this->last_name)) {
            $parts[] = preg_replace('/\s+/', '', strtolower($this->last_name));
        }
        $username = implode('.', $parts);
        $idx = 1;
        while (User::findByUsername($username)) {
            $username = implode('.', $parts) . '.' . $idx;
            $idx++;
        }
        return $username;
    }

    public function submit()
    {
        // new user
        $password = new Password($this->password);

        $bootcampUser = false;
        // first check if it's a Bootcamp user
        $user = $this->checkBootcampUser();
        if ($user instanceof User) {
            $bootcampUser = true;
            $user->username = $this->username;
            if ($this->webhookData) {
                $user->click_funnels_data = $this->webhookData;
                $user->password_salt = 'no-password';
                $user->password = 'no-password';
            }
            else {
                $user->password_salt = $password->getSalt();
                $user->password = $password->getHash();
            }
            $user->save();
        }
        else {
            $user = new User();
            $user->username = $this->username;
            $user->email = $this->email;
            if ($this->webhookData) {
                $user->click_funnels_data = $this->webhookData;
                $user->password_salt = 'no-password';
                $user->password = 'no-password';
            }
            else {
                $user->password_salt = $password->getSalt();
                $user->password = $password->getHash();
            }
            $user->user_role_id = UserRole::CLIENT;
            $user->save();
        }

        // store additional user info
        if ($this->first_name) {
            $userMeta = new UserMeta();
            $userMeta->setAttributes([
                'user_id' => $user->id,
                'key' => UserMeta::FIRST_NAME,
                'value' => $this->first_name,
                'visible' => 1
            ]);
            $userMeta->save();
            $userMeta = new UserMeta();
            $userMeta->setAttributes([
                'user_id' => $user->id,
                'key' => UserMeta::LAST_NAME,
                'value' => $this->last_name,
                'visible' => 1
            ]);
            $userMeta->save();
            $userMeta = new UserMeta();
            $userMeta->setAttributes([
                'user_id' => $user->id,
                'key' => 'address_line_1',
                'value' => $this->address,
                'visible' => 1
            ]);
            $userMeta->save();
        }

        return self::setupAccount($user, $this->password, $bootcampUser, $this);
    }

    protected function checkBootcampUser($attribute = 'email')
    {
        $bootcamp = UserPlanType::getPlanTypeObject(UserPlanType::PLAN_TYPE_BOOTCAMP);
        $bootcampAccess = UserPlanType::getPlanTypeObject(UserPlanType::PLAN_TYPE_BOOTCAMP_ACCESS);

        $user = User::find()
            ->where([
                $attribute => $this->$attribute,
                'user_role_id' => UserRole::CLIENT
            ])
            ->one();
        if ($user) {
            $userPlan = UserPlan::getActivePlan($user->id);
            if (!$userPlan ||
                ($userPlan && $userPlan->user_plan_type_id != $bootcamp->id && $userPlan->user_plan_type_id != $bootcampAccess->id)) {
                return false;
            }
            return $user;
        }
        return null;
    }

    public static function setupAccount(User $user, $password, $bootcampUser = false, Register $register = null)
    {
        // new registration
        $authToken = Yii::$app->security->generateRandomString(32) . time();

        // bootcamp users need to verify email again
        if ($bootcampUser) {
            UserRegistration::deleteAll(['user_id' => $user->id]);
        }
        $registration = new UserRegistration();
        $registration->setAttributes([
            'user_id' => $user->id,
            'auth_token' => $authToken,
            'source' => $register->source
        ]);
        $registration->save();

        if ($registration->id) {
            // assign default coach
            if (!$bootcampUser) {
                $parent = new UserParent();
                $parent->setAttributes([
                    'user_id' => $user->id,
                    'user_parent_id' => Yii::$app->params['defaultCoachId']
                ]);
                $parent->save();
            }

            // set up the initial plan if necessary
            if ($register && $register->scenario) {
                $userPlanType = UserPlanType::getPlanTypeObject($register->scenario);
                $userPlan = new UserPlan();
                $userPlan->setAttributes([
                    'user_id' => $user->id,
                    'user_plan_type_id' => $userPlanType->id,
                    'start_time' => Formatter::localDatetimeToUtcDatetime(time(), 'Y-m-d H:i:s')
                ]);
                if ($register->stripe_charge_id) {
                    $userPlan->stripe_charge_id = $register->stripe_charge_id;
                }
                if ($register->plan_variation_id) {
                    $userPlan->user_plan_variation_id = $register->plan_variation_id;
                } else {
                    $variations = $userPlanType->userPlanVariations;
                    if (count($variations) > 0) {
                        $userPlan->user_plan_variation_id = $variations[0]->id;
                    }
                }
                if (!empty($register->coupon)) {
                    $userPlan->stripe_coupon_id = $register->coupon;
                }
                if ($register->scenario == UserPlanType::PLAN_TYPE_BOOTCAMP) {
                    $userPlan->valid_until = Formatter::localDatetimeToUtcDatetime(strtotime('+' . UserPlan::BOOTCAMP_FREE_DAYS . ' days'), 'Y-m-d 23:59:59');
                }
                // handle one-off payments
                elseif (empty($userPlan->userPlanVariation->stripe_plan_id)) {
                    $planLength = $userPlan->userPlanVariation->plan_length;
                    if ($userPlan->userPlanVariation->plan_length_in_weeks) {
                        $daysMonths = 'weeks';
                    }
                    else {
                        $daysMonths = $planLength > 12 || $planLength <= 0 ? 'days' : 'months';
                    }
                    $validUntilString = $planLength != 0 ?
                        ($planLength > 0 ? '+' : '-') . abs($planLength) . ' ' . $daysMonths :
                        '+100 years'; // <-- one-time fee
                    $userPlan->valid_until = Formatter::localDatetimeToUtcDatetime(
                        strtotime($validUntilString),
                        'Y-m-d 23:59:59'
                    );
                }
                $userPlan->save();

                // not for bootcamp plan and gravity form registrations (the charge goes via API in this case)
                if ($register->scenario != UserPlanType::PLAN_TYPE_BOOTCAMP && strpos($register->scenario, 'gravity-') !== 0) {
                    // save the payment info and subscribe user if possible
                    if ($register->token && $register->plan_variation_id) {
                        try {
                            // one-off payments
                            if (empty($userPlan->userPlanVariation->stripe_plan_id)) {
                                UserPlan::createAndCharge($userPlan, $user->id, $register->token, $userPlan->userPlanVariation->plan_length > 0);
                            }
                            // subscriptions
                            else {
                                $customerId = UserPlan::createAndSubscribe($userPlan, $user->id, $register->token, $userPlan->userPlanVariation->plan_length > 0);
                                // upfront payments
                                if (!empty($userPlan->userPlanVariation->upfront_payment) && $customerId) {
                                    UserPlan::chargeOnly($userPlan, $customerId, $userPlan->userPlanVariation->upfront_payment);
                                }
                            }
                        }
                        catch (\Stripe\Error\Card $e) {
                            // error validating the card, need to cleanup everything created so far
                            $userPlan->delete();
                            if (isset($parent)) {
                                $parent->delete();
                            }
                            $registration->delete();
                            UserMeta::deleteAll(['user_id' => $user->id]);
                            $user->resetAccount(false);
                            $user->delete();
                            Yii::$app->session->setFlash('registration', [['danger', $e->getMessage()]]);
                            return false;
                        }
                    }
                }

                if ($register->scenario == UserPlanType::PLAN_TYPE_BOOTCAMP) {
                    DripHelper::sendCustomEvent('Customer Registered', $user->email, [
                        'plan' => $userPlan->userPlanType->name,
                        'option' => UserPlan::BOOTCAMP_FREE_DAYS . ' days',
                        'coupon' => $register->coupon
                    ]);
                }
                // handle one-off payments
                elseif (empty($userPlan->userPlanVariation->stripe_plan_id)) {
                    $planLength = $userPlan->userPlanVariation->plan_length;
                    if ($userPlan->userPlanVariation->plan_length_in_weeks) {
                        $daysMonths = 'weeks';
                    }
                    else {
                        $daysMonths = $planLength > 12 || $planLength <= 0 ? 'days' : 'months';
                    }
                    DripHelper::sendCustomEvent('Customer Registered', $user->email, [
                        'plan' => $userPlan->userPlanType->name,
                        'option' => abs($planLength) . ' ' . $daysMonths,
                        'coupon' => $register->coupon
                    ]);
                }
                else {
                    DripHelper::sendCustomEvent('Customer Registered', $user->email, [
                        'plan' => $userPlan->userPlanType->name,
                        'option' => $userPlan->userPlanVariation->getPricePeriod('') . 'ly',
                        'coupon' => $register->coupon
                    ]);
                }

                // register in SharpSpring
                //$user->sharpSpringRegistered($userPlan);
            }

            // add to mailchimp - not needed for now
            // $user->mailchimpSubscribe();

            // register the user on the membership site
            $user->createAndSaveMembershipUser($password, $register->first_name, $register->last_name);

            // email the user
            $authUrl = Url::toRoute(['register/confirm', 'auth_token' => $authToken], true);
            DripHelper::sendEmailEvent(
                'registration',
                $user->email,
                'Action Required: Activate Your Account',
                [
                    'username' => $user->getFullName(),
                    'auth_url' => $authUrl
                ]
            );
            if ($register->resetPasswordAfterRegistration) {
                $request = $user->createNewPasswordResetRequest($register->source);
                // first make sure the user exists in Drip
                DripHelper::addSubscriber($user->email, $user->id);
                // send email
                DripHelper::sendEmailEvent(
                    'setup',
                    $user->email,
                    'Action Required: Set Up Your Account',
                    [
                        'email' => $user->email,
                        'auth_url' => $request->getPasswordResetUrl()
                    ]
                );
            }
            return true;
        }
        return false;
    }

}
