# Getting Started

Let's take a look into Codeception architecture. We assume that you already [installed](http://codeception.com/install) it, and bootstrapped your first test suites. Codeception has generated three of them: unit, functional, and acceptance. They are well described in the previous chapter. Inside your `/tests` folder you will have three config files and three directories with names corresponding to these suites. Suites are independent groups of tests with a common purpose. 

## Guys

One of the main concepts of Codeception is representation of tests as actions of a person. We call this person a Guy. We have a CodeGuy, who executes functions and tests the code. We also have a TestGuy, a qualified tester, who tests the application as a whole, with knowledge of it's internals. And a WebGuy, a user who works with our application through an interface that we provide.

Each of these Guys are PHP classes along with the actions that they are allowed to do. As you can see, each of these Guys have different abilities. They are not constant, you can extend them. You can create new Guys If you like, but one Guy belongs to one suite. You can see the Guy classes inside the suite directories.

Guy classes are not written, they are generated by the build command:

```
$ php codecept.phar build
```

## Writing a Sample Scenario

By default tests are written as narrative scenarios. To make a php file a valid scenario, it's name should have a `Cept` suffix. 

Let's say, we created a file `tests/acceptance/SigninCept.php`

We can do that by running command:

```
$ php codecept.phar generate:cept Signin
```

``` php
<?php
$I = new WebGuy($scenario);
?>
```

A Scenario always starts with Guy class initialization. After that, writing a scenario is just like typing `$I->` and choosing a proper action from the auto-completion list.

Let's sign in to our site. We assume that we have a 'login' page where we are getting authorized by login and password. Then we are moved to a user page, where we see the text `Hello, %username%`. Let's look at how this scenario is written in Codeception.

``` php
<?php
$I = new WebGuy($scenario);
$I->wantTo('log in as regular user');
$I->amOnPage('/login');
$I->fillField('Username','davert');
$I->fillField('Password','qwerty');
$I->click('Login');
$I->see('Hello, davert');
?>
```

Before we execute this test, we should make sure that the site is running on a local web server. Open the `tests/acceptance.suite.yml` file and replace the url with the url of your web application:

``` yaml
config:
    PhpBrowser:
        url: 'http://myappurl.local'
```

If you don't have a web server running, you can use the [PHP Built-in Web Server](http://php.net/manual/en/features.commandline.webserver.php) which is available in PHP 5.4. 

After you set the proper url, you can run this test with the command:

``` bash
$ php codecept.phar run
```

In the output you should see: 

``` bash
Suite acceptance started
Trying log in as regular user (SigninCept.php) - Ok

Suite functional started

Suite unit started


Time: 1 second, Memory: 21.00Mb

OK (1 test, 1 assertions)
```

Let's get a detailed output:

```bash
$ php codecept.phar run acceptance --steps
```

We should see a step-by-step report on the performed actions.

```bash
Suite acceptance started
Trying to log in as regular user (SigninCept.php)
Scenario:
* I am on page "/login"
* I fill field "Username" "davert"
* I fill field "Password" "qwerty"
* I click "Login"
* I see "Hello, davert"
  OK


Time: 0 seconds, Memory: 21.00Mb

OK (1 test, 1 assertions)
```

That was a very simple test that you can reproduce for your own site.
By emulating the user's actions you can test all of your site the same way.

Give it a try!

## Modules and Helpers

The actions in Guy classes are taken from modules. With the `build` command described above, Codeception emulates multiple inheritance. Modules are designed to have one action performed with one method. According to the DRY principle, if you use the same scenario components in different modules, you can combine them and move them to a custom module. By default each suite has an empty module, which can extend Guy classes. They are stored in the `helpers` directory.

## Bootstrap

Each suite has it's own bootstrap file. It's located in the suite directory and is named `_bootstrap.php`. It will be executed before each test.

Write any setup preparations for the suite there.

## Tests
Codeception supports three test formats. Besides the previously described scenario-based Cept format, Codeception can also execute PHPUnit test files for unit testing, and a hybrid scenario-unit Cest format. They are covered in later chapters. There is no difference in the way the tests of either format will be run in the suite.

## Important Notes

Codeception is pretty smart when executing test scenarios. But you should keep in mind that Codeception executes each scenario two times: one for analysis and one for execution. So, any custom PHP code put in the file will be executed two times! Probably, you don't need that. Before injecting any custom php code (which is not using the `$I` object) into your test, please specify the stage when it should be executed.

__To say it again: each test file is run two times: for analyses and execution.__

For instance, this code will be run when the scenario is analyzed.

```php
<?php
if ($scenario->preload()) {
    Logger::log('analyzing');
}
?>
```

and this is executed in the test runtime:

```php
<?php
if ($scenario->running()) {
    Logger::log('running');
}
?>
```

But whenever you can try to keep you tests simple and avoid usage any custom PHP code at all. Use only the `$I` object whenever it is possible. Keep the test clean and readable.

## Configuration

Codeception has a global configuration in `codeception.yml` and a config for each suite. We support also `.dist` configuration files. If you have several developers in a project, put shared settings into `codeception.dist.yml` and personal settings into `codeception.yml`. Same goes for suite configs. For example, the `unit.suite.yml` will be merged with `unit.suite.dist.yml`. 

By default your global configuration file will be this:

```yaml
paths:
    # where the modules stored
    tests: tests

    # logs and debug 
    # outputs will be written there
    log: tests/_log

    # directory for fixture data    
    data: tests/_data

    # directory for custom modules (helpers)
    helpers: tests/_helpers

settings:

    # name of bootstrap that will be used
    # each bootstrap file should be 
    # inside a suite directory.

    bootstrap: _bootstrap.php

    # You can extend the suite class if you need to.
    suite_class: \PHPUnit_Framework_TestSuite

    # by default it's false on Windows
    # use [ANSICON](http://adoxa.110mb.com/ansicon/) to colorize output.
    colors: true

    # Tests (especially functional) can take a lot of memory
    # We set a high limit for them by default.
    memory_limit: 1024M

    # If a log should be written.
    # Every action in test is logged.
    # Logs are kept for 3 days.
    log: true

# Global modules configuration.    
modules:
    config:
        Db:
            dsn: ''
            user: ''
            password: ''
            dump: tests/_data/dump.sql
```

Suite configuration acceptance.yml

```yaml
class_name: WebGuy
modules:
    # enabled modules and helpers
    enabled:
        - PhpBrowser
        - WebHelper
        - Db

    # local module configuration. Overrides the global.        
    config:
    	Db:
            dsn:
```


## Running Tests

Tests can be started with the `run` command.

```bash
$ php codecept.phar run
```

With the first argument you can run tests from one suite.

```bash
$ php codecept.phar run acceptance
```

To run exactly one test, add a second argument. Provide a local path to the test, from the suite directory.

```bash
$ php codecept.phar run acceptance SigninCept.php
```

There are plenty of options you can use.

* `steps` - all performed actions will be printed to console.
* `debug` - additional debug output will be printed.
* `colors` - turn on colors (if disabled)
* `silent` - don't show the progress output.
* `report` - format results in report mode.
* `no-exit` - don't provide exit codes on finish. This option may be useful for using Codeception with some CI servers like Bamboo.
* `config` - specify different config file for current run.

With the following options you can set the output in the most suitable format.

* `html` - generate html file with results. It will be stored as 'report.html' in tests/_log.
* `xml` - generate report in JUnit format for CI services. It will be stored as 'report.xml' in tests/_log.
* `tap` - generate report in TAP format. It will be stored as 'report.tap.log' in tests/_log.
* `json` - generate report in Json format. It will be stored as 'report.json' in tests/_log.

Example:

```bash
$ php codecept.phar run --steps --xml --html
```

This command will run all tests for all suites, displaying the steps, and building html and xml reports.

### Generators

There are plenty of useful Codeception commands.

* `generate:cept` *suite* *filename* - Generates a sample Cept scenario.
* `generate:cest` *suite* *filename* - Generates a sample Cest test.
* `generate:test` *suite* *filename* - Generates a sample PHPUnit Test.
* `generate:suite` *suite* *guy* - Generates a new suite with the given Guy class name.
* `generate:scenarios` *suite* - Generates text files containing scenarios from tests.


## Conclusion

We took a look into the Codeception structure. Most of the things you need were already generated by the `bootstrap` command. After you have reviewed the basic concepts and configurations, you can start writing your first scenarios. 